/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * faketape test suite
 */

#include <config.h>
#include "faketape-lib.h"

#include <fcntl.h>	// O_WRONLY

#define CONFIG_CATCH_MAIN

#include <catch2/catch.hpp>

using Response = FakeTape::Response;

class temptape : public FakeTape {
	private:
		char dirnam[16] = "faketape.XXXXXX";
		std::string filename;
	public:
		temptape() {
			if (!mkdtemp(dirnam)) throw std::runtime_error("mkdtemp failed");
			filename = std::string(dirnam) + "/faketape.img";
			debug = std::make_unique<std::ostringstream>();
		}
		~temptape() {
			close();
			if(unlink(filename.c_str()) == -1)
				std::cerr << "Unlink of " << filename << " failed" << std::endl;
			if(rmdir(dirnam) == -1)
				std::cerr << "Unlink of " << dirnam << " failed" << std::endl;
			if (trace())
				CHECK(reinterpret_cast<std::ostringstream*>(debug.get())->str().size());
			else
				CHECK(reinterpret_cast<std::ostringstream*>(debug.get())->str().size() == 0);
			trace(false);
		}
		Response open(int flags) {
			return FakeTape::open(filename, flags);
		}
};

TEST_CASE("FilesCanBeCreatedOpenedAndClosed") {
	temptape faketape;
	std::vector<uint8_t> databuffer;

	auto postest = [&faketape](uint32_t faketapeo, size_t fblk, size_t blk) -> bool {
		CHECK(faketape.getFileNo() == faketapeo);
		CHECK(faketape.getFileBlock() == fblk);
		CHECK(faketape.getPos() == blk);
		return faketape.getFileNo() == faketapeo && faketape.getFileBlock() == fblk && faketape.getPos() == blk;
	};

	REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);
	REQUIRE(postest(0,0,0));
	REQUIRE(faketape.writeData((const uint8_t*)"file1", 5) == Response::OK);
	REQUIRE(postest(0,1,1));
	faketape.close();
	REQUIRE(faketape.open(O_WRONLY) == Response::OK);
	REQUIRE(postest(1,0,2));
	faketape.trace(true);
	REQUIRE(faketape.writeData((const uint8_t*)"file2", 5) == Response::OK);
	REQUIRE(postest(1,1,3));
	faketape.close();

	REQUIRE(faketape.open(O_RDONLY) == Response::OK);
	REQUIRE(faketape.rewind() == Response::OK);

	std::vector<uint8_t> data(10);
	size_t rl;

	REQUIRE(postest(0,0,0));
	REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
	REQUIRE(postest(0,1,1));
	REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
	REQUIRE(postest(1,0,2));
	REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
	REQUIRE(postest(1,1,3));
	REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
	REQUIRE(postest(2,0,4));
	faketape.close();
}

TEST_CASE("test_writeSomeData") {
	std::vector<uint8_t> databuffer;

	temptape faketape;
	REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);

	for (size_t i = 0; i<4096; ++i) {
		std::string data = std::to_string(i);
		REQUIRE(faketape.writeDataAsync(reinterpret_cast<const uint8_t*>(data.data()), data.size()) == Response::OK);
	}
	/* Write End-of-File */
	faketape.writeDataAsync(nullptr, 0);
	faketape.close();

	REQUIRE(faketape.open(O_RDONLY) == Response::OK);

	REQUIRE(faketape.rewind() == Response::OK);

	std::vector<uint8_t> data(10);
	size_t rl;
	for (size_t i = 0; i<4096; ++i) {
		REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);

		std::string want = std::to_string(i);
		std::string got = std::string(data.data(), data.data() + rl);

		REQUIRE(want == got);
	}
	REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
	REQUIRE(rl == 0);
	faketape.close();
}

TEST_CASE("test_writeTwoFiles") {
	std::vector<uint8_t> databuffer;

	temptape faketape;

	for(int l = 0; l<2; ++l) {
		REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);

		for (size_t i = 0; i<4096; ++i) {
			std::string data = std::to_string(i) + std::to_string(l);
			REQUIRE(faketape.writeDataAsync(reinterpret_cast<const uint8_t*>(data.data()), data.size()) == Response::OK);
		}
		faketape.close();
	}

	REQUIRE(faketape.open(O_RDONLY) == Response::OK);
	REQUIRE(faketape.rewind() == Response::OK);
	faketape.close();

	for(int l = 0; l<2; ++l) {
		REQUIRE(faketape.open(O_RDONLY) == Response::OK);

		std::vector<uint8_t> data(10);
		size_t rl;
		for (size_t i = 0; i<4096; ++i) {
			REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);

			std::string want = std::to_string(i) + std::to_string(l);
			std::string got = std::string(data.data(), data.data() + rl);

			REQUIRE(want == got);
		}
		REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);
		REQUIRE(rl == 0);
		faketape.close();
	}
}

TEST_CASE("test_seeking") {
	std::vector<uint8_t> databuffer;

	temptape faketape;

	for(int l = 0; l<2; ++l) {
		REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);

		for (size_t i = 0; i<4096; ++i) {
			std::string data = std::to_string(i>4097?i-4097:i) + std::to_string(l);
			REQUIRE(faketape.writeDataAsync(reinterpret_cast<const uint8_t*>(data.data()), data.size()) == Response::OK);
		}
		faketape.close();
	}

	REQUIRE(faketape.open(O_RDONLY) == Response::OK);

	REQUIRE(faketape.blockSeek(1) == Response::OK);

	for (size_t i = 1; i<8192; i+=100) {
		REQUIRE(faketape.blockSeek(i) == Response::OK);
		std::vector<uint8_t> data(10);
		size_t rl;
		REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);

		std::string want = std::to_string(i>4097?i-4097:i) + std::to_string(i>4097?1:0);
		std::string got = std::string(data.data(), data.data() + rl);

		REQUIRE(want == got);
	}

	for (size_t j = 1; j<8192; j+=100) {
		size_t i = 8191 - j;
		REQUIRE(faketape.blockSeek(i) == Response::OK);
		std::vector<uint8_t> data(10);
		size_t rl;
		REQUIRE(faketape.readData(data.data(), 10, &rl) == Response::OK);

		std::string want = std::to_string(i>4097?i-4097:i) + std::to_string(i>4097?1:0);
		std::string got = std::string(data.data(), data.data() + rl);

		REQUIRE(want == got);
	}

	faketape.close();
}

TEST_CASE("test_zerocompression") {
	std::vector<uint8_t> databuffer;

	SECTION("write and read back blocks of zero bytes") {
		constexpr std::array<size_t, 5> tests = {1, 2, 3, 4007, 1<<24};
		for(size_t sz : tests) {
			temptape faketape;
			static constexpr size_t repeat = 1100;
			std::vector<uint8_t> data(sz);

			REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);

			for (size_t i = 0; i<repeat; ++i) {
				REQUIRE(faketape.writeDataAsync(data.data(), data.size()) == Response::OK);
			}
			faketape.close();

			REQUIRE(faketape.open(O_RDONLY) == Response::OK);
			REQUIRE(faketape.rewind() == Response::OK);

			data.resize(std::min(sz+5, FakeTape::getMaxBlockSize()));
			for (size_t i = 0; i<repeat; ++i) {
				size_t rl;
				REQUIRE(faketape.readData(data.data(), data.size(), &rl) == Response::OK);
				REQUIRE(rl == sz);
				REQUIRE (std::find_if(data.data(), data.data()+sz, [](uint8_t b){ return b!=0; }) == data.data()+sz);
			}
			faketape.close();
		}
	}

	SECTION("check that blocks of zero data use no tape except for the index blocks") {
		uint8_t data[4096] = {0};

		temptape faketape;
		REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);

		for (size_t i = 0; i<4096; ++i) {
			REQUIRE(faketape.writeDataAsync(data, sizeof data) == Response::OK);
		}
		faketape.close();

		REQUIRE(faketape.open(O_RDONLY) == Response::OK);
		REQUIRE(faketape.rewind() == Response::OK);

		for (size_t i = 0; i<4096; ++i) {
			size_t rl;
			REQUIRE(faketape.readData(data, sizeof data, &rl) == Response::OK);
			faketape.sync();
			off_t p = faketape.getDiskPosForTest();
			auto calcBlockPos = [](size_t x) -> size_t {
				++x;
				if (x < FakeTape::recsInFirstBlock)
					return 1;
				x-=FakeTape::recsInFirstBlock;
				return 2 + x/FakeTape::recsPerBlock;
			};
			off_t targetp = 4096 * calcBlockPos(i);	//maxblocks is private in FakeTape
			REQUIRE(p == targetp);
		}
		faketape.close();
	}
}

TEST_CASE("CheckThatMaxBlockSizeIsEnforced") {
	temptape faketape;
	std::vector<uint8_t> databuffer;

	REQUIRE(faketape.setMaxBlockSize((1<<24)+1) == Response::ERROR);
	REQUIRE(faketape.getMaxBlockSize() == 1<<24);

	REQUIRE(faketape.setMaxBlockSize(1024) == Response::OK);
	REQUIRE(faketape.getMaxBlockSize() == 1024);

	REQUIRE(faketape.open(O_WRONLY|O_CREAT) == Response::OK);
	databuffer.resize(1025);

	REQUIRE(faketape.writeDataAsync(databuffer.data(), 1024) == Response::OK);
	REQUIRE(faketape.writeDataAsync(databuffer.data(), 1025) == Response::ERROR);
	faketape.close();

	REQUIRE(faketape.open(O_RDONLY) == Response::OK);
	REQUIRE(faketape.rewind() == Response::OK);

	size_t rl;
	REQUIRE(faketape.readData(databuffer.data(), 1025, &rl) == Response::ERROR);
	REQUIRE(faketape.readData(databuffer.data(), 1024, &rl) == Response::OK);
	REQUIRE(rl == 1024);
	REQUIRE(faketape.readData(databuffer.data(), 1024, &rl) == Response::OK);
	REQUIRE(rl == 0);
	faketape.close();

	REQUIRE(faketape.setMaxBlockSize(1<<24) == Response::OK);
	REQUIRE(faketape.getMaxBlockSize() == 1<<24);

}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
