#!/usr/bin/env python
# coding: utf-8
#
#    Project: Azimuthal integration
#             https://github.com/silx-kit/pyFAI
#
#    Copyright (C) 2015-2018 European Synchrotron Radiation Facility, Grenoble, France
#
#    Principal author:       Jérôme Kieffer (Jerome.Kieffer@ESRF.eu)
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"test suite for Geometric Refinement class"

__author__ = "Jérôme Kieffer"
__contact__ = "Jerome.Kieffer@ESRF.eu"
__license__ = "MIT"
__copyright__ = "European Synchrotron Radiation Facility, Grenoble, France"
__date__ = "16/10/2020"

import unittest
import os
import numpy
import random
import logging

from .utilstest import UtilsTest

logger = logging.getLogger(__file__)
from .. import geometryRefinement
from .. import calibrant
GeometryRefinement = geometryRefinement.GeometryRefinement


class TestGeometryRefinement(unittest.TestCase):
    """ tests geometric refinements with or without spline"""

    def test_noSpline(self):
        """tests geometric refinements without spline"""

        pixelSize = [1.5e-5, 1.5e-5]
        data = [[1585.9999996029055, 2893.9999991192408, 0.53005649383067788],
                [1853.9999932086102, 2873.0000001637909, 0.53005649383067788],
                [2163.9999987531855, 2854.9999987738884, 0.53005649383067788],
                [2699.9999977914931, 2893.9999985831755, 0.53005649383067788],
                [3186.9999966428777, 3028.9999985930604, 0.53005649383067788],
                [3595.0000039534661, 3167.0000022967461, 0.53005649383067788],
                [3835.0000007197755, 3300.0000002536408, 0.53005649383067788],
                [1252.0000026881371, 2984.0000056421914, 0.53005649383067788],
                [576.99992486352289, 3220.0000014469815, 0.53005649383067788],
                [52.999989546760531, 3531.9999975314959, 0.53005649383067788],
                [520.99999862452842, 2424.0000005943775, 0.65327673902147754],
                [1108.0000045189499, 2239.9999793751085, 0.65327673902147754],
                [2022.0000098770186, 2136.9999921020726, 0.65327673902147754],
                [2436.000002384907, 2137.0000034435734, 0.65327673902147754],
                [2797.9999973906524, 2169.9999849019205, 0.65327673902147754],
                [3516.0000041508365, 2354.0000059814265, 0.65327673902147754],
                [3870.9999995625412, 2464.9999964079757, 0.65327673902147754],
                [3735.9999952703465, 2417.9999888223151, 0.65327673902147754],
                [3374.0001428680412, 2289.9999885080188, 0.65327673902147754],
                [1709.99999872134, 2165.0000006693272, 0.65327673902147754],
                [2004.0000081015958, 1471.0000012076148, 0.7592182246175333],
                [2213.0000015244159, 1464.0000243454842, 0.7592182246175333],
                [2115.9999952456633, 1475.0000015176133, 0.7592182246175333],
                [2242.0000023736206, 1477.0000046142911, 0.7592182246175333],
                [2463.9999967564663, 1464.0000011704756, 0.7592182246175333],
                [2986.000011249705, 1540.9999994523619, 0.7592182246175333],
                [2760.0000031761901, 1514.0000002442944, 0.7592182246175333],
                [3372.0000025298395, 1617.9999995345927, 0.7592182246175333],
                [3187.0000005152106, 1564.9999952212884, 0.7592182246175333],
                [3952.0000062252166, 1765.0000234029771, 0.7592182246175333],
                [200.99999875941003, 1190.0000046393075, 0.85451320177642376],
                [463.00000674257342, 1121.9999956648539, 0.85451320177642376],
                [1455.0000001416358, 936.99999830341949, 0.85451320177642376],
                [1673.9999958962637, 927.99999934328309, 0.85451320177642376],
                [2492.0000021823594, 922.00000383122256, 0.85451320177642376],
                [2639.9999948599761, 936.00000247819059, 0.85451320177642376],
                [3476.9999490636446, 1027.9999838362451, 0.85451320177642376],
                [3638.9999965727247, 1088.0000258143732, 0.85451320177642376],
                [4002.0000051610787, 1149.9999925115812, 0.85451320177642376],
                [2296.9999822277705, 908.00000939182382, 0.85451320177642376],
                [266.00000015817864, 576.00000049157074, 0.94195419730133967],
                [364.00001493127616, 564.00000136247968, 0.94195419730133967],
                [752.99999958240187, 496.9999948653093, 0.94195419730133967],
                [845.99999758606646, 479.00000730401808, 0.94195419730133967],
                [1152.0000082161678, 421.9999937722655, 0.94195419730133967],
                [1215.0000019951258, 431.00019867504369, 0.94195419730133967],
                [1728.0000096657914, 368.00000247754218, 0.94195419730133967],
                [2095.9999932673395, 365.99999862304219, 0.94195419730133967],
                [2194.0000006543587, 356.99999967534075, 0.94195419730133967],
                [2598.0000021676074, 386.99999979901884, 0.94195419730133967],
                [2959.9998766657627, 410.00000323183838, 0.94195419730133967]]
        data = numpy.array(data, dtype=numpy.float64)
        # tth = data[:,2]
        ring = [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3,
                3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5,
                5, 5, 5, 5, 5]
        ds = [4.15695, 2.93940753, 2.4000162, 2.078475, 1.85904456,
              1.69706773, 1.46970377, 1.38565, 1.31454301, 1.25336758,
              1.2000081, 1.15293049, 1.11099162, 1.0392375, 1.00820847,
              0.97980251, 0.95366973, 0.92952228, 0.90712086, 0.88626472,
              0.84853387, 0.83139, 0.81524497, 0.8000054, 0.77192624,
              0.75895176, 0.73485188, 0.72363211, 0.71291104, 0.7026528,
              0.692825, 0.68339837, 0.67434634, 0.65727151, 0.64920652,
              0.64143131, 0.63392893, 0.62668379, 0.61968152, 0.61290884,
              0.60000405, 0.59385, 0.58788151, 0.58208943, 0.57646525,
              0.571001, 0.56568924, 0.55549581, 0.55060148, 0.54583428,
              0.54118879, 0.53224291, 0.52793318, 0.52372647, 0.51961875,
              0.51560619, 0.51168517, 0.50785227, 0.50410423, 0.50043797,
              0.49685056]  # LaB6
        wavelength = 1.54e-10
        mycalibrant = calibrant.Calibrant(dSpacing=ds, wavelength=wavelength)
        data[:, 2] = ring

        r = GeometryRefinement(data, pixel1=pixelSize[0], pixel2=pixelSize[1],
                               wavelength=wavelength, calibrant=mycalibrant)
        r.refine2(10000000)

#        ref = numpy.array([0.089652, 0.030970, 0.027668, -0.699407, 0.010067, 0.000001])
        ref = numpy.array([0.089750, 0.030897, 0.027172, -0.704730, 0.010649, 3.51e-06])
        self.assertAlmostEqual(abs(numpy.array(r.param) - ref).max(), 0.0, 2, "ref=%s obt=%s delta=%s" % (list(ref), r.param, abs(numpy.array(r.param) - ref)))

    def test_Spline(self):
        """tests geometric refinements with spline"""
        splineFine = UtilsTest.getimage("frelon.spline")
        data = [[795, 288, 0.3490658503988659],
                [890, 260, 0.3490658503988659],
                [948, 249, 0.3490658503988659],
                [710, 325, 0.3490658503988659],
                [601, 392, 0.3490658503988659],
                [1167, 248, 0.3490658503988659],
                [1200, 340, 0.3490658503988659],
                [1319, 285, 0.3490658503988659],
                [1362, 302, 0.3490658503988659],
                [1436, 338, 0.3490658503988659],
                [1526, 397, 0.3490658503988659],
                [1560, 424, 0.3490658503988659],
                [1615, 476, 0.3490658503988659],
                [1662, 529, 0.3490658503988659],
                [1742, 650, 0.3490658503988659],
                [1778, 727, 0.3490658503988659],
                [1824, 891, 0.3490658503988659],
                [1831, 947, 0.3490658503988659],
                [1832, 1063, 0.3490658503988659],
                [1828, 1106, 0.3490658503988659],
                [1828, 1106, 0.3490658503988659],
                [1810, 1202, 0.3490658503988659],
                [1775, 1307, 0.3490658503988659],
                [1724, 1407, 0.3490658503988659],
                [1655, 1502, 0.3490658503988659],
                [1489, 1649, 0.3490658503988659],
                [1397, 1700, 0.3490658503988659],
                [1251, 1752, 0.3490658503988659],
                [1126, 1772, 0.3490658503988659],
                [984, 1770, 0.3490658503988659],
                [907, 1758, 0.3490658503988659],
                [801, 1728, 0.3490658503988659],
                [696, 1681, 0.3490658503988659],
                [634, 1644, 0.3490658503988659],
                [568, 1596, 0.3490658503988659],
                [520, 1553, 0.3490658503988659],
                [453, 1479, 0.3490658503988659],
                [403, 1408, 0.3490658503988659],
                [403, 1408, 0.3490658503988659],
                [363, 1337, 0.3490658503988659],
                [320, 1228, 0.3490658503988659],
                [303, 1161, 0.3490658503988659],
                [287, 1023, 0.3490658503988659],
                [287, 993, 0.3490658503988659],
                [304, 846, 0.3490658503988659],
                [329, 758, 0.3490658503988659],
                [341, 726, 0.3490658503988659],
                [402, 606, 0.3490658503988659],
                [437, 555, 0.3490658503988659],
                [513, 467, 0.3490658503988659]
                ]
#        data = numpy.array(data)
        random.shuffle(data)
        tth = data[0][2]
        # data[:, 2] = ring
        wl = 2e-10 * numpy.sin(tth / 2.0)
        ds = [1.0]
        mycalibrant = calibrant.Calibrant(dSpacing=ds, wavelength=wl)
        r2 = GeometryRefinement(data, dist=0.1, splineFile=splineFine, wavelength=wl, calibrant=mycalibrant)
#        r2.poni1 = 5e-2
#        r2.poni2 = 5e-2
        r2.rot1_max = 0.0001
        r2.rot1_min = -0.0001
        r2.rot2_max = 0.0001
        r2.rot2_min = -0.0001
        r2.rot3_max = 0.0001
        r2.rot3_min = -0.0001
        r2.refine2(10000000, fix=("wavelength", "rot1", "rot2", "rot3"))
        ref2 = numpy.array([0.1, 4.917310e-02, 4.722438e-02, 0, 0., 0.00000])
        for i, key in enumerate(("dist", "poni1", "poni2", "rot1", "rot2", "rot3")):
            self.assertAlmostEqual(ref2[i], r2.__getattribute__(key), 1,
                                   "%s is %s, I expected %s%s%s" % (key, r2.__getattribute__(key), ref2[i], os.linesep, r2))
#        assert abs(numpy.array(r2.param) - ref2).max() < 1e-3

    def test_synthetic(self):
        """
        Synthetic dataset usd to ensure the reliability

        Calibration image produced with:
        import fabio
        from pyFAI import calibrant, AzimuthalIntegrator, detector_factory
        det = detector_factory("Fairchild")
        ai = AzimuthalIntegrator(0.1,0.05, 0.06, 0.07, 0.08, detector=det)
        LaB6 = calibrant.ALL_CALIBRANTS("LaB6")
        LaB6.wavelength=1e-10
        img = LaB6.fake_calibration_image(ai, W=0.00001)
        header = {"dist": 1.000000e-01, "poni1": 5.000000e-02,
                  "poni2": 6.000000e-02, "rot1":0.070000, "rot2": 0.080000,
                  "rot3": 0.000000, "detector": "Fairchild"}
        fabio.edfimage.EdfImage(data=img,header=).save("sample.edf")
        then calibrated manually and validate the error:
        """
        data = [[2703.99951172, 103.998390198, 3],
                [482.997772217, 3017.0, 3],
                [1751.71136302, 673.529311657, 3],
                [1175.43571347, 1279.81449679, 3],
                [2759.00024414, 81.9995117188, 3],
                [501.997741699, 2876.99951172, 3],
                [1768.68567717, 659.510243505, 3],
                [593.997680664, 2455.99951172, 3],
                [946.208278716, 1624.98937929, 3],
                [512.998474121, 4040.0, 3],
                [2561.99975586, 164.998565674, 3],
                [467.997924805, 3178.99975586, 3],
                [1958.76428576, 513.416629195, 3],
                [1933.80644698, 531.42432642, 3],
                [911.109468937, 1686.18258257, 3],
                [550.998352051, 2622.99975586, 3],
                [505.997680664, 2851.99951172, 3],
                [1236.48361963, 1200.7757373, 3],
                [1815.69851473, 621.489398181, 3],
                [536.998535156, 2686.0, 3],
                [994.289456546, 1545.02063258, 3],
                [1059.31673837, 1443.88719118, 3],
                [1951.78004602, 518.418576777, 3],
                [2422.85098103, 231.060760319, 3],
                [482.99822998, 3834.00024414, 3],
                [762.998168945, 1985.99938965, 3],
                [899.142529309, 1708.04712091, 3],
                [629.998718262, 2336.0, 3],
                [2906.99926758, 28.9980239868, 3],
                [1043.35314721, 1468.00171669, 3],
                [1719.68732107, 700.539667308, 3],
                [1674.68084416, 739.564454526, 3],
                [2127.82311094, 399.299559116, 3],
                [863.999694824, 1775.0, 3],
                [573.998779297, 2529.0, 3],
                [885.123720884, 1733.96129294, 3],
                [694.998779297, 2150.99951172, 3],
                [616.997741699, 2377.99926758, 3],
                [485.99810791, 3858.99975586, 3],
                [489.99786377, 3890.0, 3],
                [1894.71161973, 560.449989676, 3],
                [459.00289917, 3479.00024414, 3],
                [764.999145508, 1980.99975586, 3],
                [667.998779297, 2224.0, 3],
                [2961.99926758, 10.9980697632, 3],
                [596.998291016, 2444.99951172, 3],
                [465.99786377, 3210.0, 3],
                [600.998535156, 2431.0, 3],
                [539.998413086, 2671.99975586, 3],
                [534.99798584, 2695.99951172, 3],
                [790.998291016, 1923.99951172, 3],
                [507.997589111, 2839.99975586, 3],
                [1712.67906186, 706.540253967, 3],
                [1800.68304923, 633.481516778, 3],
                [1172.4323253, 1283.81848672, 3],
                [1104.35309201, 1377.88289711, 3],
                [498.998168945, 3953.99975586, 3],
                [1099.40161508, 1384.96775111, 3],
                [1665.66401288, 747.561096013, 3],
                [499.997528076, 3960.00024414, 3],
                [582.998840332, 2495.00024414, 3],
                [481.998077393, 3825.00024414, 3],
                [612.997924805, 2390.99926758, 3],
                [2395.73875621, 245.088028431, 3],
                [786.999694824, 1931.99987793, 3],
                [555.997924805, 2601.99951172, 3],
                [655.998840332, 2258.0, 3],
                [1582.64706171, 823.604478598, 3],
                [986.277669907, 1558.02046429, 3],
                [757.998901367, 1996.99975586, 3],
                [540.997802734, 2667.99951172, 3],
                [1001.26428205, 1533.8970595, 3],
                [2200.78819269, 354.269370139, 3],
                [478.997619629, 3797.00024414, 3],
                [2420.86883987, 232.042865217, 3],
                [2399.0, 243.999771118, 3],
                [481.997619629, 3025.99975586, 3],
                [717.999694824, 2092.00024414, 3],
                [1190.42495084, 1259.87731857, 3],
                [1214.46922582, 1228.77899235, 3],
                [473.99798584, 3104.00024414, 3],
                [607.998840332, 2406.99975586, 3],
                [503.99810791, 2864.00024414, 3],
                [2158.01945025, 380.257992387, 3],
                [1090.39694989, 1397.97668305, 3],
                [1041.37568355, 1471.02638916, 3],
                [779.998291016, 1947.99926758, 3],
                [1116.38490093, 1360.84228638, 3],
                [1040.34532434, 1472.82265328, 3],
                [515.998474121, 2792.99975586, 3],
                [718.998657227, 2089.99926758, 3],
                [761.998657227, 1987.99975586, 3],
                [807.047155619, 1887.92891639, 3],
                [461.997680664, 3292.0, 3],
                [462.002655029, 3580.00024414, 3],
                [659.99786377, 2246.99926758, 3],
                [460.002868652, 3524.99975586, 3],
                [2942.99926758, 16.998468399, 3],
                [469.997802734, 3151.99926758, 3],
                [1560.62788677, 844.603595167, 3],
                [488.998382568, 3883.0, 3],
                [1518.06903539, 186.269275635, 4],
                [127.539863944, 2304.90279071, 4],
                [2.00088214874, 3807.99975586, 4],
                [736.35367915, 1018.06610837, 4],
                [180.001358032, 2127.00073242, 4],
                [19.0012493134, 3948.99951172, 4],
                [95.0012283325, 2428.00048828, 4],
                [474.668508112, 1439.96332825, 4],
                [191.692563951, 2093.91765522, 4],
                [14.0006713867, 2867.0, 4],
                [1061.1554033, 615.118878104, 4],
                [581.325944155, 1254.00592441, 4],
                [45.0006752014, 2665.99975586, 4],
                [957.202497765, 733.110474803, 4],
                [1001.17358591, 682.106410928, 4],
                [775.242806017, 964.049660545, 4],
                [1378.07832093, 304.197022974, 4],
                [39.0013313293, 4077.99975586, 4],
                [1409.06329362, 277.198698223, 4],
                [713.271178156, 1051.04615553, 4],
                [179.296277523, 2130.92503364, 4],
                [228.598861635, 1987.91869316, 4],
                [226.001296997, 1993.00073242, 4],
                [1320.10595678, 356.197222248, 4],
                [751.294508249, 997.068496056, 4],
                [118.81592387, 2337.89715467, 4],
                [362.236587629, 1663.93954866, 4],
                [400.441100538, 1583.97183819, 4],
                [0.0, 3788.0, 4],
                [1744.95644668, 16.2645991445, 4],
                [903.248565972, 798.107018709, 4],
                [318.848430306, 1759.97940915, 4],
                [8.00124645233, 3863.0, 4],
                [256.698118627, 1912.94729704, 4],
                [24.0013160706, 3983.99951172, 4],
                [1116.160906, 556.150645718, 4],
                [715.329612792, 1048.06117878, 4],
                [1667.98685064, 71.2514461279, 4],
                [32.0005493164, 4035.0, 4],
                [28.000875473, 4010.0, 4],
                [1041.16325112, 637.116560593, 4],
                [22.0010700226, 3969.99951172, 4],
                [79.0009994507, 2497.00024414, 4],
                [541.349159092, 1321.01110603, 4],
                [1523.06696247, 182.25386411, 4],
                [16.0011615753, 2851.0, 4],
                [1609.00357667, 115.238092765, 4],
                [111.001312256, 2364.00048828, 4],
                [1485.03883456, 213.212132454, 4],
                [41.001209259, 2688.0, 4],
                [1.00077486038, 3798.0, 4],
                [105.905757129, 2387.90158176, 4],
                [1140.15200995, 531.154803351, 4],
                [11.001203537, 2890.00048828, 4],
                [100.929146051, 2407.9014383, 4],
                [81.0011062622, 2488.00048828, 4],
                [852.276487261, 862.095621236, 4],
                [97.9464449286, 2419.91502638, 4],
                [813.216446757, 913.036101177, 4],
                [578.324245721, 1258.98668865, 4],
                [638.343123794, 1163.04305071, 4],
                [128.803521991, 2300.90338317, 4],
                [891.196771845, 813.06628684, 4],
                [339.487590253, 1713.97336781, 4],
                [315.537430733, 1767.01963923, 4],
                [18.0006790161, 3941.00024414, 4],
                [1630.04490116, 99.2706407607, 4],
                [1402.07983333, 283.207578763, 4],
                [27.001039505, 2774.00048828, 4],
                [43.0008659363, 2677.00024414, 4],
                [1638.04691935, 93.2772672176, 4],
                [67.0004653931, 2552.99975586, 4],
                [620.490099818, 1191.02382993, 4],
                [819.227353349, 905.05951025, 4],
                [136.906114638, 2271.91999496, 4],
                [566.324590355, 1278.9480034, 4],
                [553.112632662, 1300.95822861, 4],
                [20.0009403229, 2822.00024414, 4],
                [1468.05785288, 227.218342677, 4],
                [25.0007019043, 3990.00024414, 4],
                [883.1930632, 823.054901291, 4],
                [183.717290044, 2117.94321888, 4],
                [168.745391965, 2164.92482182, 4],
                [1622.04961035, 105.272948056, 4],
                [188.707058787, 2102.91901361, 4],
                [987.195629269, 698.118389614, 4],
                [926.217438087, 770.105636008, 4],
                [584.328950316, 1249.02678554, 4],
                [1437.07004081, 253.214575619, 4],
                [77.0009994507, 2506.00024414, 4],
                [402.444198936, 1579.95467908, 4],
                [253.796579465, 1918.97998174, 4],
                [1462.05798138, 232.216169432, 4],
                [80.00050354, 2493.00024414, 4],
                [975.175339505, 712.092769995, 4],
                [23.0009346008, 2801.00024414, 4],
                [129.872851312, 2296.92537253, 4],
                [2.00091838837, 2969.99975586, 4],
                [1331.09756678, 346.191923112, 4],
                [1084.17177871, 590.14431861, 4],
                [807.211707309, 921.02168371, 4],
                [3312.99975586, 512.000488281, 2],
                [3177.00048828, 547.000549316, 2],
                [1552.29128715, 1677.0450879, 2],
                [1339.18368658, 2012.98972228, 2],
                [986.001403809, 3144.00024414, 2],
                [3708.99975586, 447.001342773, 2],
                [1952.09925586, 1230.20555714, 2],
                [1289.00183105, 2108.00097656, 2],
                [2973.82735325, 612.847960055, 2],
                [2169.01207923, 1049.29927737, 2],
                [1346.41739428, 1999.97709346, 2],
                [1840.15840018, 1338.17109594, 2],
                [2234.01251232, 1001.33453012, 2],
                [986.001037598, 3771.00024414, 2],
                [1221.19546102, 2253.97347354, 2],
                [2652.88805231, 749.587563396, 2],
                [3719.99975586, 446.00112915, 2],
                [1776.19436711, 1405.14839539, 2],
                [984.001098633, 3164.0, 2],
                [1022.00115967, 4015.99975586, 2],
                [2922.85558698, 631.675208807, 2],
                [3420.0, 489.001068115, 2],
                [3867.99951172, 436.001159668, 2],
                [1007.00134277, 2985.0, 2],
                [1360.36055562, 1974.97941732, 2],
                [3828.00024414, 438.001098633, 2],
                [3689.0, 449.001312256, 2],
                [971.001342773, 3365.00024414, 2],
                [3118.80313008, 564.957009435, 2],
                [1466.36112791, 1801.03056236, 2],
                [996.001037598, 3854.99951172, 2],
                [1028.00146484, 4046.99975586, 2],
                [2267.99164841, 977.350668877, 2],
                [3780.99951172, 441.001098633, 2],
                [1646.23439096, 1555.07123364, 2],
                [1470.34503958, 1795.01740419, 2],
                [3390.00048828, 495.001098633, 2],
                [1482.33664083, 1777.01240411, 2],
                [1963.08702826, 1220.20566931, 2],
                [1026.00158691, 2878.00024414, 2],
                [2130.04664512, 1079.28705901, 2],
                [3400.00048828, 493.000976562, 2],
                [2401.96764392, 889.18811357, 2],
                [973.001098633, 3315.00024414, 2],
                [1659.2421357, 1539.09822563, 2],
                [2460.92491273, 853.563950598, 2],
                [3251.00024414, 527.001037598, 2],
                [3928.0, 434.001251221, 2],
                [1939.10049625, 1242.1975408, 2],
                [1425.97948822, 1864.02971989, 2],
                [1295.32946759, 2095.11858893, 2],
                [1254.0755427, 2180.99166497, 2],
                [3515.00073242, 472.001342773, 2],
                [2825.97780383, 669.846126601, 2],
                [2399.97748368, 890.416900218, 2],
                [2484.92454185, 839.561943293, 2],
                [2834.85620061, 666.702821851, 2],
                [3213.0, 537.000915527, 2],
                [3052.88054482, 585.449228168, 2],
                [1808.17052977, 1371.15577155, 2],
                [2876.8900814, 649.464025438, 2],
                [990.000915527, 3807.0, 2],
                [970.001281738, 3508.99975586, 2],
                [2851.85789372, 659.685542762, 2],
                [975.001098633, 3637.99951172, 2],
                [1633.28230813, 1571.09249736, 2],
                [1729.19629896, 1457.11268844, 2],
                [1522.21269932, 1719.01036369, 2],
                [3528.0, 470.000946045, 2],
                [1436.05766935, 1848.01875849, 2],
                [2640.89025332, 755.573986471, 2],
                [1045.93165588, 2788.89541233, 2],
                [976.001037598, 3653.99951172, 2],
                [1721.22968769, 1466.12813751, 2],
                [1263.55868345, 2160.07040174, 2],
                [971.001403809, 3551.0, 2],
                [3274.99975586, 521.000976562, 2],
                [982.001403809, 3731.00024414, 2],
                [980.001342773, 3707.99951172, 2],
                [3102.84163015, 569.706816316, 2],
                [1626.23699161, 1580.03690225, 2],
                [1077.07232607, 2661.96421124, 2],
                [3022.85170798, 595.665097833, 2],
                [2906.84196858, 637.801264822, 2],
                [979.001403809, 3220.0, 2],
                [2806.91686761, 678.308028847, 2],
                [3025.86277337, 594.588669896, 2],
                [3731.00024414, 445.001190186, 2],
                [2550.92921476, 802.538493037, 2],
                [2139.03530635, 1072.2886292, 2],
                [2664.9022961, 743.607547164, 2],
                [1303.17609084, 2080.98609644, 2],
                [1179.54051912, 2354.94901167, 2],
                [3560.99975586, 465.000915527, 2],
                [1202.94928032, 2297.87026191, 2],
                [1439.35232353, 1842.99093573, 2],
                [3115.83445854, 565.753395081, 2],
                [1092.46465215, 2608.93134064, 2],
                [2293.99706542, 959.401011229, 2],
                [2530.94090599, 813.505604923, 2],
                [1696.20536886, 1495.08669695, 2],
                [759.103856936, 200.098927185, 5],
                [838.090231799, 117.105425887, 5],
                [373.108510658, 675.01294115, 5],
                [254.22653465, 851.048441105, 5],
                [196.112120003, 943.97402318, 5],
                [648.090467513, 324.061813872, 5],
                [125.41529128, 1064.02726791, 5],
                [635.131745607, 339.090328172, 5],
                [501.09088704, 503.028840141, 5],
                [205.210369319, 929.038089294, 5],
                [789.097563535, 168.100615121, 5],
                [347.129298091, 712.023128621, 5],
                [454.200472325, 564.079917952, 5],
                [275.067146897, 818.964798298, 5],
                [329.160652161, 738.045691397, 5],
                [624.093098454, 352.058980085, 5],
                [139.153739557, 1039.98655412, 5],
                [747.094688207, 213.088334076, 5],
                [179.153750807, 971.997937873, 5],
                [641.135731056, 332.09417063, 5],
                [736.090229981, 225.081845418, 5],
                [220.130790874, 904.992319388, 5],
                [439.201152816, 584.077126853, 5],
                [325.121398248, 744.011540636, 5],
                [582.133901194, 402.080162711, 5],
                [153.117273696, 1015.96663908, 5],
                [177.335606903, 975.041295204, 5],
                [464.108749293, 551.03477177, 5],
                [417.098586492, 614.015897781, 5],
                [949.06570933, 7.10693087429, 5],
                [383.089184046, 661.001032041, 5],
                [716.092079364, 247.079004727, 5],
                [103.984483234, 1101.93297345, 5],
                [276.285343409, 817.05765152, 5],
                [33.2340170145, 1234.0154937, 5],
                [473.188954026, 539.0806465, 5],
                [218.19425112, 908.037244696, 5],
                [135.135496065, 1046.97352916, 5],
                [204.076886728, 930.955716804, 5],
                [690.102213413, 276.082230307, 5],
                [147.201318622, 1026.02778709, 5],
                [356.151416034, 699.044301111, 5],
                [406.118987717, 629.029658157, 5],
                [151.304533184, 1019.03746679, 5],
                [185.160783798, 962.004728405, 5],
                [31.2111070007, 1237.99411883, 5],
                [494.088398375, 512.025167653, 5],
                [208.310942173, 924.04653706, 5],
                [809.092969343, 147.101139754, 5],
                [102.307130307, 1105.03037268, 5],
                [592.109799176, 390.066136599, 5],
                [164.209568411, 997.031747006, 5],
                [12.4942139387, 1275.00337564, 5],
                [160.144440562, 1003.98590421, 5],
                [242.141449347, 870.005641221, 5],
                [682.115728095, 285.090444289, 5],
                [258.139574945, 845.008675901, 5],
                [251.103620641, 855.981266428, 5],
                [477.098541133, 534.029339857, 5],
                [480.125888884, 530.053313226, 5],
                [90.1818842143, 1126.99229503, 5],
                [0.0, 1299.0, 5],
                [698.108118415, 267.088204987, 5],
                [76.0270170793, 1152.93225654, 5],
                [261.345854133, 840.054481041, 5],
                [385.150444955, 658.051190443, 5],
                [237.101012208, 877.97670472, 5],
                [183.344009191, 965.041782215, 5],
                [144.332164228, 1031.03617312, 5],
                [167.186642155, 992.021065315, 5],
                [100.138182402, 1108.96594819, 5],
                [78.1872270554, 1148.99206549, 5],
                [559.165881917, 430.090837181, 5],
                [9.46362581849, 1281.00761227, 5],
                [323.098617382, 746.994445097, 5],
                [497.130122066, 508.061470378, 5],
                [142.071410581, 1034.94129838, 5],
                [59.5178416371, 1184.00373977, 5],
                [408.243316755, 626.079470575, 5],
                [448.183488831, 572.073914677, 5],
                [87.5768757463, 1131.99345748, 5],
                [358.235660881, 696.068009429, 5],
                [28.1670467854, 1243.96272366, 5],
                [629.14196229, 346.094829865, 5],
                [282.181034058, 808.044157133, 5],
                [71.1803465486, 1161.9849252, 5],
                [46.2220247239, 1209.00956609, 5],
                [515.13879475, 485.069636524, 5],
                [233.226354778, 884.044905476, 5],
                [133.412427515, 1050.02868766, 5],
                [597.109625086, 384.066914454, 5],
                [247.181633234, 862.038749959, 5],
                [675.095641755, 293.072777063, 5],
                [338.132619888, 725.023787681, 5],
                [519.116066106, 480.054499868, 5],
                [148.476235032, 1024.02096598, 5],
                [587.11562866, 396.070330538, 5],
                [18.6960722208, 1262.96020179, 5],
                [607.133463129, 372.085128479, 5],
                [577.16752845, 408.095665209, 5],
                [306.14584434, 772.026290761, 5],
                [3843.0, 1851.00085449, 0],
                [2290.00097656, 3890.99951172, 0],
                [2381.01813314, 2859.01382001, 0],
                [2324.9446761, 4014.11995341, 0],
                [3622.0, 1875.00097656, 0],
                [2247.00097656, 3463.0, 0],
                [2366.19211894, 2894.99056321, 0],
                [4000.0, 1852.00073242, 0],
                [2508.45351428, 2618.95461858, 0],
                [3241.0109064, 1990.97219195, 0],
                [3946.99975586, 1850.00073242, 0],
                [3979.0, 1851.00073242, 0],
                [3639.92606127, 1872.54308498, 0],
                [2914.02063424, 2182.17811017, 0],
                [2351.00146484, 4091.99926758, 0],
                [4017.00024414, 1853.00073242, 0],
                [3875.0, 1850.00073242, 0],
                [3427.05968637, 1921.34576535, 0],
                [2302.45439363, 3084.0516867, 0],
                [3611.00024414, 1877.00097656, 0],
                [4067.0, 1857.00024414, 0],
                [2372.9214751, 2878.02428203, 0],
                [2305.35691106, 3073.06404106, 0],
                [2273.57014236, 3212.02331612, 0],
                [2246.00292969, 3521.99975586, 0],
                [2267.80744517, 3251.93964226, 0],
                [3083.03357735, 2070.77548026, 0],
                [2339.00170898, 4058.99951172, 0],
                [2388.17416692, 2841.12232918, 0],
                [2258.00097656, 3316.00024414, 0],
                [3310.98408164, 1962.1263552, 0],
                [2626.10048845, 2457.04084374, 0],
                [3327.04686644, 1955.52889013, 0],
                [3571.02346897, 1884.54711267, 0],
                [3700.99975586, 1863.00061035, 0],
                [2284.09393794, 3864.02115541, 0],
                [2859.03601142, 2225.15254615, 0],
                [3755.00024414, 1857.00073242, 0],
                [2333.00146484, 2984.00073242, 0],
                [2660.09246433, 2417.05802435, 0],
                [2248.00073242, 3437.00024414, 0],
                [2340.83958134, 4063.98161517, 0],
                [2252.00097656, 3375.00048828, 0],
                [2341.7119711, 2960.91689683, 0],
                [3073.96849031, 2076.3005462, 0],
                [3168.99951172, 2024.99914551, 0],
                [2676.09570645, 2399.07351193, 0],
                [2364.19403638, 2899.98999478, 0],
                [3350.90610167, 1947.58666873, 0],
                [2287.00048828, 3877.99902344, 0],
                [2417.56667322, 2779.07784286, 0],
                [2418.51101884, 2777.08518112, 0],
                [2724.1059904, 2348.12269144, 0],
                [2268.62550598, 3245.94431398, 0],
                [3270.91015112, 1978.54133517, 0],
                [2988.98837512, 2129.2530919, 0],
                [2255.00073242, 3343.0, 0],
                [2304.24288945, 3079.97589117, 0],
                [3521.96699289, 1896.16017491, 0],
                [2307.60377866, 3957.06154961, 0],
                [2884.02827586, 2205.15974374, 0],
                [3085.86567716, 2069.62685961, 0],
                [2290.00097656, 3135.00024414, 0],
                [3041.04511989, 2095.63787103, 0],
                [3686.00024414, 1865.00036621, 0],
                [2273.24655679, 3809.03496547, 0],
                [2313.74455327, 3978.08259609, 0],
                [2288.50429302, 3140.03845973, 0],
                [2605.11478142, 2483.03924209, 0],
                [2248.00097656, 3590.0, 0],
                [2498.1600794, 2635.01848538, 0],
                [2410.35705754, 2794.97319449, 0],
                [2250.00097656, 3624.99975586, 0],
                [3161.07908992, 2028.19946033, 0],
                [2460.37862122, 2698.96953591, 0],
                [2700.08818224, 2373.08364324, 0],
                [2247.00097656, 3564.0, 0],
                [2276.71719831, 3829.99315458, 0],
                [2496.60595691, 2637.91112856, 0],
                [3367.87531917, 1941.72758019, 0],
                [3167.09522904, 2025.00403708, 0],
                [2263.90195805, 3748.06369229, 0],
                [2874.01738175, 2213.09488707, 0],
                [3362.00048828, 1943.00146484, 0],
                [2587.96908777, 2505.04781073, 0],
                [3115.01426717, 2052.99731742, 0],
                [2936.02650039, 2165.92159714, 0],
                [3348.00024414, 1948.0012207, 0],
                [2285.00073242, 3157.00024414, 0],
                [2275.94849366, 3205.91938016, 0],
                [2564.18548524, 2537.03171605, 0],
                [3099.05956903, 2061.44538689, 0],
                [2556.36753359, 2547.98748167, 0],
                [3525.93275724, 1895.46024823, 0],
                [3580.00048828, 1883.00097656, 0],
                [2396.8509562, 2823.04103296, 0],
                [3092.81715333, 2065.76146859, 0],
                [2320.93556701, 4003.99759827, 0],
                [3051.9266981, 2089.44386414, 0],
                [2456.76243293, 2705.86960129, 0],
                [2272.00073242, 3803.99951172, 0],
                [3847.0, 1080.00170898, 1],
                [1582.99804688, 3042.99975586, 1],
                [2640.96045366, 1466.6165652, 1],
                [2637.93751588, 1468.55726942, 1],
                [3528.99951172, 1115.99816895, 1],
                [1543.99743652, 3597.0, 1],
                [1556.9979248, 3214.99975586, 1],
                [2648.86666584, 1461.44405466, 1],
                [2300.70239472, 1733.65697798, 1],
                [1633.99755859, 2827.99902344, 1],
                [2091.57219064, 1956.8127576, 1],
                [3365.99951172, 1150.99853516, 1],
                [1551.99829102, 3712.00024414, 1],
                [1542.99755859, 3401.99975586, 1],
                [1562.99768066, 3166.99975586, 1],
                [3315.99975586, 1163.99926758, 1],
                [3161.99902344, 1211.99865723, 1],
                [2891.04989029, 1324.71703964, 1],
                [2771.83238693, 1387.3148824, 1],
                [1936.56059223, 2166.9525625, 1],
                [3546.0, 1112.99829102, 1],
                [2887.06409098, 1326.56213006, 1],
                [2154.3693516, 1884.1282814, 1],
                [3147.99926758, 1216.99816895, 1],
                [2429.7634674, 1620.57532614, 1],
                [1607.99890137, 2925.99975586, 1],
                [1869.5189288, 2274.9976902, 1],
                [1888.58146936, 2242.99498304, 1],
                [3661.99975586, 1095.99804688, 1],
                [1761.6460093, 2482.07716533, 1],
                [1602.99755859, 2947.99926758, 1],
                [1824.10266799, 2355.91978176, 1],
                [1568.99816895, 3124.99951172, 1],
                [1701.99951172, 2624.0, 1],
                [2849.83512565, 1345.24932194, 1],
                [1578.99768066, 3909.00048828, 1],
                [2054.54890999, 2002.8390567, 1],
                [2893.00573876, 1323.95021672, 1],
                [1542.00341797, 3567.0, 1],
                [3357.99975586, 1152.99865723, 1],
                [1545.99804688, 3634.99975586, 1],
                [4006.99975586, 1078.9979248, 1],
                [1583.99780273, 3936.00048828, 1],
                [3127.99975586, 1223.99926758, 1],
                [3688.0, 1092.99780273, 1],
                [1541.99755859, 3438.0, 1],
                [1695.99890137, 2639.99951172, 1],
                [3334.99975586, 1158.99853516, 1],
                [1775.99768066, 2452.99902344, 1],
                [2920.07523978, 1310.59326547, 1],
                [3301.99975586, 1167.99853516, 1],
                [1578.9979248, 3064.99975586, 1],
                [1942.51217526, 2157.92688228, 1],
                [3705.99975586, 1090.99829102, 1],
                [1541.0032959, 3526.99951172, 1],
                [3142.0, 1218.99890137, 1],
                [1565.99804688, 3829.00048828, 1],
                [2478.84386277, 1581.57949814, 1],
                [3820.0, 1081.99816895, 1],
                [1613.99829102, 2901.99975586, 1],
                [1544.99780273, 3356.99951172, 1],
                [2694.95646639, 1432.52343091, 1],
                [1560.9979248, 3181.99975586, 1],
                [1798.23969012, 2405.94399119, 1],
                [2645.81950668, 1463.41655147, 1],
                [1837.21515886, 2332.92852935, 1],
                [3485.99975586, 1123.99865723, 1],
                [3021.02952312, 1265.15600151, 1],
                [1552.99829102, 3251.99975586, 1],
                [1786.28394222, 2429.98162481, 1],
                [4083.99975586, 1081.99768066, 1],
                [1543.99768066, 3376.99975586, 1],
                [3839.99975586, 1080.99804688, 1],
                [1653.99890137, 2760.99975586, 1],
                [1816.28421903, 2370.93447856, 1],
                [3803.00024414, 1082.99829102, 1],
                [2534.86817759, 1539.55391404, 1],
                [1604.99841309, 2939.0, 1],
                [1809.4609282, 2384.04467283, 1],
                [2922.0, 1310.0, 1],
                [1605.99780273, 2934.99951172, 1],
                [3266.99975586, 1177.99951172, 1],
                [1728.17016423, 2557.16471268, 1],
                [1592.99804688, 3981.0, 1],
                [3670.00024414, 1094.99829102, 1],
                [1832.32915866, 2340.93063179, 1],
                [3057.87655091, 1250.05375654, 1],
                [2933.03251249, 1304.31320739, 1],
                [1650.99816895, 2770.99951172, 1],
                [1972.46699107, 2113.86367784, 1],
                [3411.99951172, 1139.99780273, 1],
                [1571.99804688, 3105.99975586, 1],
                [1598.99841309, 2965.00024414, 1],
                [1561.99841309, 3173.99975586, 1],
                [2839.97388294, 1350.37556511, 1],
                [1620.0, 4095.0, 1],
                [1801.29406404, 2399.95554309, 1],
                [3167.99926758, 1209.99829102, 1],
                [1914.61959648, 2200.98170173, 1],
                [1585.99841309, 3026.99926758, 1],
                [1552.99804688, 3722.00024414, 1]]
        mycalibrant = calibrant.ALL_CALIBRANTS("LaB6")
        mycalibrant.wavelength = 1e-10
        r2 = GeometryRefinement(data, calibrant=mycalibrant, detector="Fairchild",
                                wavelength=mycalibrant.wavelength)
        print(r2)
        r2.guess_poni()
        print(r2)
        r2.refine2(10000000, fix=[])
        ref = {"dist": (0.1, 1e-5),  # value, tolerance
               "poni1": (0.05, 1e-5),
               "rot2": (0.08, 1e-4),
               "rot3": (0.0, 1e-3),
               "poni2": (0.06, 1e-5),
               "rot1": (0.07, 1e-4),
               "wavelength": (1e-10, 1e-10)}
        print(r2)
        for key in ref.keys():
            self.assertAlmostEqual(ref[key][0], r2.__getattribute__(key), delta=ref[key][1],
                                   msg="%s is %s, I expected %s%s%s" % (key, r2.__getattribute__(key), ref[key], os.linesep, r2))


def suite():
    testsuite = unittest.TestSuite()
    loader = unittest.defaultTestLoader.loadTestsFromTestCase
    testsuite.addTest(loader(TestGeometryRefinement))
    return testsuite


if __name__ == '__main__':
    runner = unittest.TextTestRunner()
    runner.run(suite())
