// -*- Mode: Go; indent-tabs-mode: t -*-

/*
 * Copyright (C) 2016 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

// Package metautil takes care of basic details of working with snap metadata formats.
package metautil

import (
	"fmt"
)

// NormalizeValue validates values and returns a normalized version of it
// (map[any]any is turned into map[string]any)
func NormalizeValue(v any) (any, error) {
	switch x := v.(type) {
	case string:
		return x, nil
	case bool:
		return x, nil
	case int:
		return int64(x), nil
	case int64:
		return x, nil
	case float64:
		return x, nil
	case float32:
		return float64(x), nil
	case []any:
		l := make([]any, len(x))
		for i, el := range x {
			el, err := NormalizeValue(el)
			if err != nil {
				return nil, err
			}
			l[i] = el
		}
		return l, nil
	case map[any]any:
		m := make(map[string]any, len(x))
		for k, item := range x {
			kStr, ok := k.(string)
			if !ok {
				return nil, fmt.Errorf("non-string key: %v", k)
			}
			item, err := NormalizeValue(item)
			if err != nil {
				return nil, err
			}
			m[kStr] = item
		}
		return m, nil
	case map[string]any:
		m := make(map[string]any, len(x))
		for k, item := range x {
			item, err := NormalizeValue(item)
			if err != nil {
				return nil, err
			}
			m[k] = item
		}
		return m, nil
	default:
		return nil, fmt.Errorf("invalid scalar: %v", v)
	}
}
