/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/run_test_suite_traversal.cpp
 *
 * @brief [LEVEL: beta] Implementation @ref
 * diagnostics::unittest::Run_Test_Suite_Traversal class
 *
 * $Id: run_test_suite_traversal.cpp,v 1.22 2005/06/23 09:54:26 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @todo test and document
 */

#include <diagnostics/unittest/test_system/run_test_suite_traversal.hpp>

#include <diagnostics/frame/logging_config.hpp>
#include <diagnostics/frame/logging_facility.hpp>

#include <diagnostics/unittest/test_case.hpp>
#include <diagnostics/unittest/test_exception.hpp>
#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/unittest/test_system/test_logger.hpp>
#include <diagnostics/unittest/name_separators.hpp>

#include <diagnostics/unittest/test_system/filter_test_data.hpp>
#include <diagnostics/unittest/test_system/mutexed_test_data.hpp>

#include <diagnostics/unittest/test_system/current_test_logger.hpp>

#include <exception>


/**
 * @brief basis for all logging -- adds the empty string for basefile
 * and file and 0 for line and sends calls @ref
 * diagnostics::Logging_Facility::log.
 *
 * @param LEVEL must be of type @ref diagnostics::Level_t
 * @param TYPE must of type @ref diagnostics::Type_t
 * @param NR_WHAT must be convertible to int (numerical field which is uninterpreted by the frame)
 * @param STR_WHAT must be convertible to ::std::string (string field which is uninterpreted by the frame)
 */
#define DIAGNOSTICS_ANONYMOUS_LOG(LEVEL,TYPE,NR_WHAT,STR_WHAT) \
::DIAGNOSTICS_NAMESPACE::Logging_Facility::log( \
(LEVEL), \
(TYPE),  \
(NR_WHAT), \
(STR_WHAT), \
"", \
"", \
0)


DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

Run_Test_Suite_Traversal::~Run_Test_Suite_Traversal()
{
}

Run_Test_Suite_Traversal::Run_Test_Suite_Traversal(Level_t const build_level,
						   Test_Data & test_data,
						   Test_Run_Results_t & results) 
    : m_build_level(build_level),
      m_test_data(test_data),
      m_results(results)
{
}

/*
void Run_Test_Suite_Traversal::enter_hook(Test_Suite const & test_suite,
					  ::std::string const & mask,
					  Level_t const level)
{
    // creating the logger, storing it and placing it in the Current_Test_Logger. 
    internal::Current_Test_Logger::set_logger(m_logger=new Test_Logger(m_build_level,m_build_level,m_results));

    // registering the logger
    Logging_Config::register_logger(m_logger);
}

void Run_Test_Suite_Traversal::exit_hook(Test_Suite const & test_suite,
					 ::std::string const & mask,
					 Level_t const level,
					 bool const exception_going_through)
{
    // unregister the logger -- testing is done
    Logging_Config::unregister_logger(m_logger);
    // also remove it from the Current_Test_Logger -- to prevent incorrect usage. 
    internal::Current_Test_Logger::set_logger(NULL);

    delete m_logger;
}
*/

void Run_Test_Suite_Traversal::visit_hook(Test_Case const & test_case,
					  Path_t const & path,
					  Level_t const level)
{
    ::std::string test_case_path;
    Path_t::const_iterator current(path.begin());
    Path_t::const_iterator const end(path.end());
    for(;current!=end;++current) test_case_path+= (*current) + DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR;
    test_case_path+=test_case.name();
    ::std::string const test_data_filter(test_case_path+DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
	
    Filter_Test_Data filter_test_data;
    filter_test_data.attach(test_data_filter,&m_test_data);
    Mutexed_Test_Data test_data;
    test_data.attach(&filter_test_data);
	
    // determining the target_level, i.e., the lowest testable level.
    Level_t target_level(LEVEL_AUDIT);
    if(test_case.is_testable_at(LEVEL_DEBUG)) target_level=LEVEL_DEBUG;
    if(test_case.is_testable_at(LEVEL_PROD)) target_level=LEVEL_PROD;
	
    // creating the logger, storing it and placing it in the
    // Current_Test_Logger and registering it
    Test_Logger logger(m_build_level,target_level,m_results);
    internal::Current_Test_Logger::push_logger(&logger);
	
    DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"CASE=\""+test_case_path+"\"");
    try {
		test_case.run(test_data);
    }
    catch(Test_System_Exception & e){
		DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_TESTCASE_EXIT,0,"CASE=\""+test_case_path+"\"");
		internal::Current_Test_Logger::pop_logger();
		throw; // ok, that's our problem -- handled at a higher level
    }
    catch(Exception & e) {
		// It is an error to end a test case with an exception. 
		// We know the type, we report.
		DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_UNEXPECTED_EXCEPTION,0,
							 "at the end of CASE=\"" + test_case_path
							 + " EXCEPTION=\"" + e.name()
							 + "\" WHAT=\"" + e.what()
							 + "\"");
    }
    catch(::std::exception & e) {
		// It is an error to end a test case with an exception. 
		// We know the type, we report.
		DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_UNEXPECTED_EXCEPTION,0,
							 "at the end of CASE=\"" + test_case_path
							 + " ::std::exception"
							 + " WHAT=\"" + e.what()
							 + "\"");
    }
    catch(...){
		// It is an error to end a test case with an exception.
		// However, it some custom exception, we do not know about.
		DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_UNEXPECTED_EXCEPTION,0,
							 "at the end of CASE=\"" + test_case_path+"\"");
    }
    DIAGNOSTICS_ANONYMOUS_LOG(LEVEL_TEST,TYPE_TESTCASE_EXIT,0,"CASE=\""+test_case_path+"\"");
    internal::Current_Test_Logger::pop_logger();
}


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;
// vim:ts=4:sw=4
