/*
 * File    : PersistentSettings.java
 * Created : 28-jun-2002 16:20
 * By      : fbusquets
 *
 * JClic - Authoring and playing system for educational activities
 *
 * Copyright (C) 2000 - 2005 Francesc Busquets & Departament
 * d'Educacio de la Generalitat de Catalunya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (see the LICENSE file).
 */

package edu.xtec.util;

import javax.swing.JOptionPane;
import java.util.Properties;
import java.util.StringTokenizer;
import java.io.*;

/**
 *
 * @author Francesc Busquets (fbusquets@xtec.net)
 * @version 1.0
 */
public class PersistentSettings {

    public static final String BASE=".edu.xtec.properties";
    public static final String BASE_OLD="edu.xtec.properties";
    public static final String BUNDLE="messages.PersistentPathsMessages";

    private static File baseFile=null;
    
    private PersistentSettings(){
    }
    
    public static File getBaseFile() throws Exception{
        
        if(baseFile==null){
            StringBuffer sb=new StringBuffer(300);
            sb.append(System.getProperty("user.dir"));
            sb.append(File.pathSeparator);
            sb.append(System.getProperty("user.home"));
            sb.append(File.pathSeparator);
            sb.append(System.getProperty("java.home")).append(File.separator).append("lib");
            sb.append(File.pathSeparator).append(System.getProperty("java.ext.dirs"));
            
            /*
            sb.append(System.getProperty("java.home")).append(File.separator).append("lib");
            sb.append(File.pathSeparator).append(System.getProperty("java.ext.dirs"));
            sb.append(File.pathSeparator).append(System.getProperty("user.home"));
            sb.append(File.pathSeparator).append(System.getProperty("user.dir"));
             */
            
            StringTokenizer st=new StringTokenizer(sb.substring(0), File.pathSeparator);
            while(baseFile==null && st.hasMoreTokens()){
                String s=st.nextToken();
                File dir=new File(s);
                if(dir.exists() && dir.isDirectory()){                    
                    File testFile=new File(dir, BASE);
                    if(testFile.canRead()){
                        baseFile=testFile;
                    }
                    else{
                        File oldTestFile=new File(dir, BASE_OLD);
                        if(oldTestFile.canRead()){
                            try{
                                if(oldTestFile.renameTo(testFile)){
                                    baseFile=testFile;
                                }                                
                            } catch(Exception ex){
                                // unable to rename!
                            }
                        }
                        /*
                        if(baseFile==null){
                            try{
                                FileOutputStream fos=new FileOutputStream(testFile);
                                fos.close();
                                baseFile=testFile;
                            } catch(Exception ex){
                                // test failed
                            }
                        }
                         */
                    }
                }
            }
            
            if(baseFile==null){
                baseFile=new File(System.getProperty("user.home"), BASE);
                System.out.println("Creating file "+baseFile);
                saveSettings(new Properties());
            }
            
            /*
            if(baseFile==null)
                throw new Exception("Unable to find or create basic properties file!");
             */
        }
        
        return baseFile;       
    }
    
    public static Properties getProperties() throws Exception{
        Properties prop=new Properties();
        InputStream is=new FileInputStream(getBaseFile());
        prop.load(is);
        is.close();
        return prop;
    }
    
    public static File getBasePathTo(String programName, Options options) throws Exception{        
        
        Properties prop=getProperties();
        
        String path=(String)prop.get(programName);
        if(path!=null){
            File d=new File(path);
            if(!d.exists() || !d.isDirectory()){                
                path=null;
            }
        }
        if(path==null){            
            String sPath=getSystemProgramsPath(options)+File.separator+programName;
            Messages messages=getMessages(options, BUNDLE);
            String msg=messages.get("cl_alert");
            int i=msg.indexOf('$');
            if(i>=0){
                msg=msg.substring(0, i)+programName+msg.substring(i+1);
            }
            boolean done=false;
            File d=new File(sPath);
            while(!done){                
                String result=(String)JOptionPane.showInputDialog(
                options.getMainComponent(), 
                msg, 
                messages.get("cl_prompt_title"),
                JOptionPane.QUESTION_MESSAGE,
                null,
                null,
                sPath);
                
                if(result==null)
                    throw new Exception("bad user input!");
                                
                d=new File(result);
                try{
                    if(!d.exists() || !d.isDirectory())
                        d.mkdirs();
                    done=true;
                } catch(Exception ex){
                    messages.showErrorWarning(options.getMainComponent(), "cl_err_unableToCreateDir", ex);
                }
            }
            
            path=d.getAbsolutePath();
            prop.setProperty(programName, path);
            saveSettings(prop);
        }        
        return new File(path);
    }
    
    private static void saveSettings(Properties prop) throws Exception{
        OutputStream out=new FileOutputStream(baseFile);
        prop.store(out, null);
        out.close();        
    }
    
    private static final String[] PRG_LOCATIONS=new String[]{
        "C:\\Archivos de programa", // Spanish
        "C:\\Programmi",            // Italian
        "C:\\Program",              // Swedish
        "C:\\Programme",            // Deutsch
        "C:\\Programas",            // Portuguese
        "C:\\Programfiler",         // Norwegian
        "C:\\Program files"         // Default
    };
    public static String getSystemProgramsPath(Options options){
        String result=System.getProperty("user.home");        
        File f=null;
        if(options.getBoolean(Options.WIN)){
            for(int i=0; i<PRG_LOCATIONS.length; i++){
                f=new File(PRG_LOCATIONS[i]);
                if(f.exists() && f.isDirectory()){
                    result=f.getAbsolutePath();
                    break;
                }
            }
        }
        else{
            if(
            ((f=new File("/usr/local")).exists() && f.isDirectory() && f.canWrite()) ||
            ((f=new File("/usr")).exists() && f.isDirectory() && f.canWrite()))
                result=f.getAbsolutePath();            
        }                
        return result;
    }
    
    public static Messages getMessages(Options options, String bundle) throws Exception{
        Messages msg=(Messages)options.get(Messages.MESSAGES);
        if(msg==null){            
            boolean mustSave=false;
            String l=(String)options.get(Messages.LANGUAGE);
            Properties prop=null;
            if(l==null){
                prop=getProperties();                
                l=prop.getProperty(Messages.LANGUAGE);
                if(l!=null)
                    options.put(Messages.LANGUAGE, l);
                else
                    mustSave=true;
            }
            else{
                options.put(Options.LANGUAGE_BY_PARAM, Options.BTRUE);
            }
            msg=Messages.getMessages(options, bundle);
            if(mustSave){
                prop.setProperty(Messages.LANGUAGE, (String)options.get(Messages.LANGUAGE));
                saveSettings(prop);                 
            }
        }
        else if (bundle!=null)
            msg.addBundle(bundle);
        
        return msg;
    }
    
    public static String getFilePath(String programName, String fileName, Options options){
        File base=null;
        try{
            base=getBasePathTo(programName, options);
        } catch(Exception ex){
            base=new File(System.getProperty("user.home"));
        }
        
        File testFile=new File(base, fileName);
        if(!testFile.canRead()){
            try{
                FileOutputStream fos=new FileOutputStream(testFile);
                fos.close();
                testFile.delete();
            } catch(Exception ex){
                testFile=new File(fileName);
            }
        }
        return testFile.getAbsolutePath();
    }                
}
