/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_vo_data.h
    \brief  LCMAPS module for creating and accessing VO data structures
    \author Martijn Steenbakkers for the EU DataGrid.

    The interface is composed of:
    -# lcmaps_createVoData(): create a VoData structure
    -# lcmaps_deleteVoData(): delete a VoData structure
    -# lcmaps_cleanVoData(): clean a VoData structure
    -# lcmaps_copyVoData(): copy (the contents of) a VoData structure
    -# lcmaps_printVoData(): print the contents of a VoData structure
    -# lcmaps_stringVoData(): cast a VoData structure into a string
    -# lcmaps_createVoMapping(): create a VoMapping structure
    -# lcmaps_deleteVoMapping(): delete a VoMapping structure
    -# lcmaps_cleanVoMapping(): clean a VoMapping structure
    -# lcmaps_copyVoMapping(): copy (the contents of) a VoMapping structure
    -# lcmaps_printVoMapping(): print the contents of a VoMapping structure
    \ingroup APIforLcmapsPlugins
*/

#ifndef LCMAPS_VO_DATA_H
#define LCMAPS_VO_DATA_H

/******************************************************************************
                             Define constants
******************************************************************************/
#define LCMAPS_NO_GID (gid_t)(-1) /*!< Invalid GID */

/******************************************************************************
                             Include header files
******************************************************************************/
#include <grp.h>

/******************************************************************************
                               Type definitions
******************************************************************************/

/*!
    \struct lcmaps_vo_data_s
    \brief structure that contains the VO information found
           in the user's proxy certificate
*/
/*!
    \typedef lcmaps_vo_data_t
    \brief Type of VO information structure
*/
typedef struct lcmaps_vo_data_s
{
    char * vo;         /*!< name of the VO to which the user belongs */
    char * group;      /*!< group within the VO */
    char * subgroup;   /*!< subgroup name */
    char * role;       /*!< the user's role */
    char * capability; /*!< the user's capability */
} lcmaps_vo_data_t;

/*!
    \struct lcmaps_vo_mapping_s
    \brief structure that contains the VO information string (or FQAN) and the
           local Gid it is mapped to.
*/
/*!
    \typedef lcmaps_vo_mapping_t
    \brief Type of VO mapping structure
*/
typedef struct lcmaps_vo_mapping_s
{
    char * vostring;   /*!< VO information string */
    char * groupname;  /*!< groupname the VO information string should be mapped to (according to the
                            groupmapfile) */
    gid_t  gid;        /*!< GID the VO information string should be mapped to (according to the
                            groupmapfile) */
} lcmaps_vo_mapping_t;

/******************************************************************************
 *                 Module definition
 *****************************************************************************/ 

/*
 * VO data section
 */
extern lcmaps_vo_data_t *
lcmaps_createVoData(
    const char * vo,
    const char * group,
    const char * subgroup,
    const char * role,
    const char * capability
);

extern int
lcmaps_deleteVoData(
    lcmaps_vo_data_t ** vo_data
);

extern int
lcmaps_cleanVoData(
    lcmaps_vo_data_t * vo_data
);

extern int
lcmaps_copyVoData(
    lcmaps_vo_data_t * dst_vo_data,
    const lcmaps_vo_data_t * src_vo_data
);

extern int
lcmaps_printVoData(
    int debug_level,
    const lcmaps_vo_data_t * vo_data
);

extern int
lcmaps_stringVoData(
    const lcmaps_vo_data_t * vo_data,
    char * buffer,
    int nchars
);


/*
 * VO group mapping section
 */
extern lcmaps_vo_mapping_t *
lcmaps_createVoMapping(
    const char * vo_data_string,
    const char * groupname,
    const gid_t  gid
);

extern int
lcmaps_deleteVoMapping(
    lcmaps_vo_mapping_t ** vo_mapping
);

extern int
lcmaps_cleanVoMapping(
    lcmaps_vo_mapping_t * vo_mapping
);

extern int
lcmaps_copyVoMapping(
    lcmaps_vo_mapping_t * dst_vo_mapping,
    const lcmaps_vo_mapping_t * src_vo_mapping
);

extern int
lcmaps_printVoMapping(
    int debug_level,
    const lcmaps_vo_mapping_t * vo_mapping
);

#endif /* LCMAPS_VO_DATA_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/interface/lcmaps_vo_data.h,v $
    $Date: 2011-03-04 21:55:13 +0100 (Fri, 04 Mar 2011) $
    $Revision: 14868 $
    $Author: dennisvd $
******************************************************************************/
