/* See LICENSE for licensing and NOTICE for copyright. */
package org.ldaptive.transport;

import java.nio.charset.StandardCharsets;
import org.ldaptive.LdapAttribute;
import org.ldaptive.LdapEntry;
import org.ldaptive.Message;
import org.ldaptive.ResultCode;
import org.ldaptive.asn1.DefaultDERBuffer;
import org.ldaptive.extended.ExtendedResponse;
import org.testng.Assert;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * Unit test for {@link ResponseParser}.
 *
 * @author  Middleware Services
 */
public class ResponseParserTest
{


  /**
   * Parser test data.
   *
   * @return  response test data
   */
  @DataProvider(name = "response")
  public Object[][] createData()
  {
    return
      new Object[][] {
        new Object[] {
          // search result entry
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b, 0x02, 0x01, 0x01, 0x64, (byte) 0x82, 0x01, 0x44, 0x04, 0x21, 0x75, 0x69,
            0x64, 0x3d, 0x38, 0x31, 0x38, 0x30, 0x33, 0x37, 0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70, 0x6c, 0x65,
            0x2c, 0x64, 0x63, 0x3d, 0x76, 0x74, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x64, 0x75, 0x30, (byte) 0x82, 0x01, 0x1d,
            0x30, 0x23, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x14, 0x04,
            0x12, 0x76, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x69, 0x61, 0x54, 0x65, 0x63, 0x68, 0x50, 0x65, 0x72, 0x73, 0x6f,
            0x6e, 0x30, 0x12, 0x04, 0x05, 0x75, 0x75, 0x70, 0x69, 0x64, 0x31, 0x09, 0x04, 0x07, 0x64, 0x66, 0x69, 0x73,
            0x68, 0x65, 0x72, 0x30, 0x0f, 0x04, 0x03, 0x75, 0x69, 0x64, 0x31, 0x08, 0x04, 0x06, 0x38, 0x31, 0x38, 0x30,
            0x33, 0x37, 0x30, 0x17, 0x04, 0x02, 0x63, 0x6e, 0x31, 0x11, 0x04, 0x0f, 0x44, 0x61, 0x6e, 0x69, 0x65, 0x6c,
            0x20, 0x57, 0x20, 0x46, 0x69, 0x73, 0x68, 0x65, 0x72, 0x30, (byte) 0x81, (byte) 0xb7, 0x04, 0x13, 0x73,
            0x75, 0x70, 0x70, 0x72, 0x65, 0x73, 0x73, 0x65, 0x64, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
            0x31, (byte) 0x81, (byte) 0x9f, 0x04, 0x14, 0x6d, 0x61, 0x69, 0x6c, 0x50, 0x72, 0x65, 0x66, 0x65, 0x72,
            0x72, 0x65, 0x64, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04, 0x14, 0x75, 0x73, 0x65, 0x72, 0x53, 0x4d,
            0x49, 0x4d, 0x45, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x04, 0x04, 0x6d, 0x61,
            0x69, 0x6c, 0x04, 0x0b, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x63, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x04, 0x12, 0x6c,
            0x6f, 0x63, 0x61, 0x6c, 0x50, 0x6f, 0x73, 0x74, 0x61, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04,
            0x15, 0x6d, 0x61, 0x69, 0x6c, 0x46, 0x6f, 0x72, 0x77, 0x61, 0x72, 0x64, 0x69, 0x6e, 0x67, 0x41, 0x64, 0x64,
            0x72, 0x65, 0x73, 0x73, 0x04, 0x09, 0x68, 0x6f, 0x6d, 0x65, 0x50, 0x68, 0x6f, 0x6e, 0x65, 0x04, 0x09, 0x6d,
            0x61, 0x69, 0x6c, 0x41, 0x6c, 0x69, 0x61, 0x73, 0x04, 0x11, 0x68, 0x6f, 0x6d, 0x65, 0x50, 0x6f, 0x73, 0x74,
            0x61, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04, 0x0a, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x50, 0x68,
            0x6f, 0x6e, 0x65},
          LdapEntry.builder()
            .messageID(1)
            .dn("uid=818037,ou=People,dc=vt,dc=edu")
            .attributes(
              new LdapAttribute("objectClass", "virginiaTechPerson"),
              new LdapAttribute("uupid", "dfisher"),
              new LdapAttribute("uid", "818037"),
              new LdapAttribute("uid", "818037"),
              new LdapAttribute("cn", "Daniel W Fisher"),
              new LdapAttribute(
                "suppressedAttribute",
                "mailPreferredAddress",
                "userSMIMECertificate",
                "mail",
                "mailAccount",
                "localPostalAddress",
                "mailForwardingAddress",
                "homePhone",
                "mailAlias",
                "homePostalAddress",
                "localPhone")).build(),
        },
        new Object[] {
          // whoami extended response
          new byte[] {
            0x30, 0x3A, 0x02, 0x01, 0x03, 0x78, 0x35, 0x0a, 0x01, 0x00, 0x04, 0x00, 0x04, 0x00, (byte) 0x8b, 0x2c, 0x64,
            0x6e, 0x3a, 0x63, 0x6e, 0x3d, 0x4a, 0x6f, 0x68, 0x6e, 0x20, 0x51, 0x75, 0x69, 0x6e, 0x63, 0x79, 0x20, 0x41,
            0x64, 0x61, 0x6d, 0x73, 0x2c, 0x6f, 0x75, 0x3d, 0x74, 0x65, 0x73, 0x74, 0x2c, 0x64, 0x63, 0x3d, 0x76, 0x74,
            0x2c, 0x64, 0x63, 0x3d, 0x65, 0x64, 0x75},
          ExtendedResponse.builder().messageID(3)
            .resultCode(ResultCode.SUCCESS)
            .matchedDN("")
            .diagnosticMessage("")
            .responseValue("dn:cn=John Quincy Adams,ou=test,dc=vt,dc=edu".getBytes(StandardCharsets.UTF_8)).build(),
        },
      };
  }


  /**
   * @param  berValue  to parse.
   * @param  response  expected response.
   *
   * @throws  Exception  On test failure.
   */
  @Test(groups = "transport", dataProvider = "response")
  public void parse(final byte[] berValue, final Message response)
    throws Exception
  {
    final ResponseParser parser = new ResponseParser();
    Assert.assertEquals(parser.parse(new DefaultDERBuffer(berValue)).get(), response);
  }
}
