//
//  Lynkeos
//  $Id: MyImageListItem.m,v 1.10 2005/01/27 23:08:13 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Tue Sep 30, 2003.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#ifdef GNUSTEP
#else
#include <QuickTime/Movies.h>
#endif

#import "MyImageListItem.h"

#define K_URL_KEY	@"url"
#define K_SELECTED_KEY	@"selected"
#define K_TIME_KEY	@"time"
#define K_INDEX_KEY	@"index"
#define K_IMAGES_KEY	@"images"
#define K_SEARCH_ORIGIN_KEY @"search"
#define K_ALIGN_OFFSET_KEY @"align"
#define K_QUALITY_KEY	@"quality"

static const MyIntegerPoint nowhere = {0,0};

@interface MyImageListItem(private)
- (void) setSelectionState :(int)state;	// Tri state option
- (void) setName :(NSString*)name ;
@end

@implementation MyImageListItem

// Initializations
- (id) init
{
   self = [super init];

   _itemURL = nil;
   _itemName = nil;
   _selection_state = 1;
   
   _hasSearchSquare = NO;
   _searchSquareOrigin = nowhere;
   _aligned = NO;
   _alignOffset = NSMakePoint(0.0,0.0);;
   _quality = -1.0;
   
   return( self );
}

- (id) initWithURL :(NSURL*)url
{
   self = [self init];
   
   _itemURL = [url retain];
   
   _itemName = [[NSFileManager defaultManager] displayNameAtPath:
                                                              [_itemURL path]];
   [_itemName retain];
   
   return( self );
}

- (void) dealloc
{
   [_itemURL release];
   [_itemName release];
   [super dealloc];
}

// Coding
- (void)encodeWithCoder:(NSCoder *)encoder
{
   [encoder encodeObject:_itemURL forKey:K_URL_KEY];
   [encoder encodeBool:(_selection_state == NSOnState) forKey:K_SELECTED_KEY];
   if ( _hasSearchSquare )
      [encoder encodePoint: NSPointFromIntegerPoint(_searchSquareOrigin) 
                    forKey:K_SEARCH_ORIGIN_KEY];
   if ( _aligned )
      [encoder encodePoint: _alignOffset forKey:K_ALIGN_OFFSET_KEY];
   [encoder encodeDouble: _quality forKey:K_QUALITY_KEY];
}

- (id)initWithCoder:(NSCoder *)decoder	// This is also an initialization
{
   if ( [decoder containsValueForKey:K_URL_KEY] &&
        [decoder containsValueForKey:K_SELECTED_KEY] )
   {
      [self initWithURL:[decoder decodeObjectForKey:K_URL_KEY]];
      [self setSelected: [decoder decodeBoolForKey:K_SELECTED_KEY]];
      if ( [decoder containsValueForKey:K_SEARCH_ORIGIN_KEY] )
         [self setSearchSquareOrigin: MyIntegerPointFromNSPoint(
                             [decoder decodePointForKey:K_SEARCH_ORIGIN_KEY])];
      if ( [decoder containsValueForKey:K_ALIGN_OFFSET_KEY] )
         [self setAlignOffset: [decoder decodePointForKey:K_ALIGN_OFFSET_KEY]];
      if ( [decoder containsValueForKey:K_QUALITY_KEY] )
         [self setQuality: [decoder decodeDoubleForKey:K_QUALITY_KEY]];
      return( self );
   }
   else
   {
      [self release];
      return( nil );
   }
}

// Accessors
- (NSURL*) getURL { return( _itemURL ); }

- (int) getSelectionState { return( _selection_state ); }

- (NSString*)getName { return( _itemName ); }

- (bool) hasSearchSquare { return( _hasSearchSquare ); }

- (MyIntegerPoint)searchSquareOrigin { return( _searchSquareOrigin ); }

- (bool)isAligned { return( _aligned ); }

- (NSPoint)alignOffset
{
   NSPoint origin = {0.0,0.0};
   if ( _aligned )
      return( _alignOffset );
   else
      return( origin );
}

- (double) getQuality { return( _quality ); }

- (NSNumber*) getIndex { return( nil ); }

- (NSImage*) getImage { return( nil ); }

- (NSSize) imageSize { return( NSMakeSize(0.0,0.0) ); }

- (void) setSelectionState :(int)state 
{
   _selection_state = state;
}

- (void) setSelected :(bool)value
{
   _selection_state = value ? NSOnState : NSOffState;
}

- (void) invalidateSearchSquare
{
   _hasSearchSquare = NO;
}

- (void) setSearchSquareOrigin :(MyIntegerPoint)origin
{
   _searchSquareOrigin = origin;
   _hasSearchSquare = YES;
}

- (void)invalidateAlignment
{
   _aligned = NO;
}

- (void) setAlignOffset :(NSPoint)offset
{
   _alignOffset = offset;
   _aligned = YES;
}

- (void) setQuality :(double)quality
{
   _quality = quality;
}

- (void) setName :(NSString*)name
{
   if ( _itemName != nil )
      [_itemName release];
   _itemName = name;
   [_itemName retain];
}

+ (id) imageListItemWithURL :(NSURL*)url
{
  return( [[[self alloc] initWithURL:url] autorelease] );
}
@end

@implementation MyImage

- (NSImage*) getImage 
{
#ifdef GNUSTEP
   return( [[[NSImage alloc] initWithContentsOfURL: [self getURL]] autorelease] );
#else
   return( [[[NSImage alloc] initByReferencingURL: [self getURL]] 
           autorelease] );
#endif
}

- (NSSize) imageSize
{
   NSImage *image = [self getImage];
   NSImageRep *rep = [image bestRepresentationForDevice:nil];
   
   return( NSMakeSize([rep pixelsWide],[rep pixelsHigh]) );
}

@end

@implementation MyMovieImage
// Initializations
- (id) initWithURL :(NSURL*)url
{
   NSAssert( NO, @"MyMovieImage doesn't support initWithURL" );
   [self release];
   return( nil );
}

- (id) initWithTime :(TimeValue)time parent:(id)up index:(long)index
{
   [self init];
   
   _parent = up;
   _time = time;
   _index = index;
   if ( _parent != nil )
      [self setName:[_parent getName]];
   
   return( self );
}

// Coding
- (void)encodeWithCoder:(NSCoder *)encoder
{
   [encoder encodeInt32:_time forKey:K_TIME_KEY];
   [encoder encodeInt32:_index forKey:K_INDEX_KEY];
   [encoder encodeBool:([self getSelectionState] == NSOnState) 
                forKey:K_SELECTED_KEY];
   if ( _hasSearchSquare )
      [encoder encodePoint: NSPointFromIntegerPoint(_searchSquareOrigin) 
                    forKey:K_SEARCH_ORIGIN_KEY];
   if ( _aligned )
      [encoder encodePoint: _alignOffset forKey:K_ALIGN_OFFSET_KEY];
   [encoder encodeDouble: _quality forKey:K_QUALITY_KEY];
}

- (id)initWithCoder:(NSCoder *)decoder
{
   if ( [decoder containsValueForKey:K_TIME_KEY] &&
        [decoder containsValueForKey:K_INDEX_KEY] &&
        [decoder containsValueForKey:K_SELECTED_KEY] )
   {
      [self initWithTime:[decoder decodeInt32ForKey:K_TIME_KEY]
                  parent:nil
                   index:[decoder decodeInt32ForKey:K_INDEX_KEY]];
      
      [self setSelected: [decoder decodeBoolForKey:K_SELECTED_KEY]];
      if ( [decoder containsValueForKey:K_SEARCH_ORIGIN_KEY] )
         [self setSearchSquareOrigin: MyIntegerPointFromNSPoint(
                             [decoder decodePointForKey:K_SEARCH_ORIGIN_KEY])];
      if ( [decoder containsValueForKey:K_ALIGN_OFFSET_KEY] )
         [self setAlignOffset: [decoder decodePointForKey:K_ALIGN_OFFSET_KEY]];
      if ( [decoder containsValueForKey:K_QUALITY_KEY] )
         [self setQuality: [decoder decodeDoubleForKey:K_QUALITY_KEY]];
      
      return( self );
   }
   else
   {
      [self release];
      return( nil );
   }
}

// Accessors
- (TimeValue) getTime { return( _time ); }

- (NSNumber*) getIndex
{
   return( [NSNumber numberWithInt:_index] );
}

- (id) getParent  { return( _parent ); }

- (NSImage*) getImage
{
#ifdef GNUSTEP
   return( [[_parent getMovie] getImageAt: _time] );
#else
   Movie qt;
   PicHandle qd;
   NSData* imageData;

   // Quicktime operations are not thread safe
   [_parent lock];

   qt = [[_parent getMovie] QTMovie];
   qd = GetMoviePict( qt, _time );
   imageData = [NSData dataWithBytes:*qd length:GetHandleSize((Handle)qd)];
   KillPicture( qd );

   [_parent unlock];

   return( [[[NSImage alloc] initWithData: imageData] autorelease] );
#endif
}

- (void) setParent :(id)parent
{
   NSAssert( _parent == nil, @"Try to override _parent movie" );
   _parent = parent;
   [self setName:[_parent getName]];
}

- (NSSize) imageSize
{
   return( [_parent imageSize] );
}

- (void) setSelected :(bool)value
{
   // Perform it
   [super setSelected:value];
   
   // And inform the parent
   [_parent imageSelectionChanged];
}

// Creator
+ (id) movieImageWithTime :(TimeValue)time parent:(id)up index:(long)index
{
   return( [[[self alloc] initWithTime:time parent:up index:index] 
           autorelease] );
}
@end

@implementation MyMovie : MyImageListItem

// Initialization
- (id) init
{
   if ( (self = [super init]) != nil )
      _qtLock = [[NSLock alloc] init];

   return( self );
}

- (id) initWithURL :(NSURL*)url
{
#ifdef GNUSTEP
   TimeValue t;
   long index;
   
   [super initWithURL :url];

   _movie = [[NSMovie alloc] initWithURL: url byReference: NO];

   if ( _movie == nil )
   {
      [self release];
      self = nil;
      return (self);
   }

   _images = [NSMutableArray array];

   // Ouvrir le film et extraire les temps des images
   t = 0;
   index = 0;
   do
   {
     t = [_movie getNextFrameTime];
      
     if ( t != -1 ) {
       [_images addObject: 
                  [MyMovieImage movieImageWithTime:t parent:self index:index]];
       index++;
     }
   } while( t != -1 );

   [_movie retain];
   [_images retain];
   
   return( self );

#else
   static const OSType vue = VisualMediaCharacteristic;
   Movie qt;
   TimeValue t;
   long index;
   
   self = [super initWithURL :url];
   
   _movie = [[NSMovie alloc] initWithURL: url byReference: NO];
   
   _images = [NSMutableArray array];
   
   // Open the movie and extract the time of each frame
   qt = [_movie QTMovie];
   t = 0;
   index = 0;
   do
   {
      GetMovieNextInterestingTime( qt, nextTimeMediaSample, 1, &vue, t, 1, 
                                   &t, NULL );
      
      if ( t != -1 )
      {
         [_images addObject: 
            [MyMovieImage movieImageWithTime:t parent:self index:index]];
         index++;
      }
   } while( t != -1 );

   [_images retain];

   if ( [_images count] == 0 )
   {
      [self release];
      self = nil;
   }

   return( self );
#endif
}

- (void) dealloc
{
   [_qtLock release];
   [_images release];
   [_movie release];

   [super dealloc];
}

// Locking
- (void) lock { [_qtLock lock]; }
- (void) unlock { [_qtLock unlock]; }

// Coding
- (void)encodeWithCoder:(NSCoder *)encoder
{
   [encoder encodeObject:[self getURL] forKey:K_URL_KEY];
   [encoder encodeObject:_images forKey:K_IMAGES_KEY];
}

- (id)initWithCoder:(NSCoder *)decoder
{
   if ( [decoder containsValueForKey:K_URL_KEY] &&
        [decoder containsValueForKey:K_IMAGES_KEY] )
   {
      NSEnumerator *iter;
      MyMovieImage *item;
      
      [super initWithURL:[decoder decodeObjectForKey:K_URL_KEY]];
      _movie = [[NSMovie alloc] initWithURL:[self getURL] byReference: NO];
      _images = [[decoder decodeObjectForKey:K_IMAGES_KEY] retain];

      iter = [_images objectEnumerator];
      while ( (item = [iter nextObject]) != nil )
         [item setParent:self];

      [self imageSelectionChanged];
      
      return( self );
   }
   else
   {
      [self release];
      return( nil );
   }
}

// Accessors
- (bool) hasSearchSquare
{
   NSAssert( NO, @"MyMovie doesn't implement hasSearchSquare" );
   return( NO );
}

- (MyIntegerPoint)searchSquareOrigin
{
   NSAssert( NO, @"MyMovie doesn't implement searchSquareOrigin" );
   return( nowhere );
}

- (bool)isAligned
{
   NSAssert( NO, @"MyMovie doesn't implement isAligned" );
   return( NO );
}

- (NSPoint)alignOffset
{
   NSAssert( NO, @"MyMovie doesn't support alignOffset" );
   return( NSMakePoint(0.0,0.0) );
}

- (NSMovie*) getMovie
{
   return( _movie );
}

- (NSSize) imageSize
{
#ifdef GNUSTEP
   return([_movie imageSize]);
#else
   Rect r;
   
   GetMovieBox( [_movie QTMovie], &r );
   
   return( NSMakeSize(r.right - r.left,r.bottom - r.top) );
#endif
}

- (int) numberOfImages
{
   return [_images count];
}

- (int) indexOfImage :(MyMovieImage*)item
{
   return( [_images indexOfObject:item] );
}

- (MyMovieImage*) getMovieImageAtIndex :(int)index
{
   return( [_images objectAtIndex:index] );
}

- (NSImage*) getImageAtIndex :(int)index
{
   return( [[_images objectAtIndex:index] getImage] );
}

- (void) addMovieImage :(MyMovieImage*)item 
{
   NSEnumerator *iter = [_images objectEnumerator];
   MyMovieImage *image;
   TimeValue itemTime = [item getTime];
   int index = 0;

   // Look for the first image whose time is after than this one
   while ( (image = [iter nextObject]) != nil &&
           [image getTime] < itemTime )
      index++;
   
   // And insert this one before it
   [_images insertObject:item atIndex:index];
}

- (void) deleteMovieImage :(MyMovieImage*)item
{
   NSAssert( [_images containsObject:item], 
             @"Cannot delete a nonexistent movie image!" );
   [_images removeObject:item];
   [self imageSelectionChanged];
}

- (void) setSelected :(bool)value
{
   NSEnumerator* list = [_images objectEnumerator];
   id item;
   
   // Propagate that state on all the images
   while ( (item = [list nextObject]) != nil )
      [item setSelected:value];
   
   [super setSelected:value];
}

- (void) invalidateSearchSquare
{
   NSEnumerator* list = [_images objectEnumerator];
   id item;
   
   // Invalidate on all the images
   while ( (item = [list nextObject]) != nil )
      [item invalidateSearchSquare];
}

- (void) setSearchSquareOrigin :(MyIntegerPoint)origin
{
   NSAssert( NO, @"MyMovie doesn't implement setSearchSquareOrigin" );
}

- (void)invalidateAlignment
{
   NSEnumerator* list = [_images objectEnumerator];
   id item;
   
   // Invalidate on all the images
   while ( (item = [list nextObject]) != nil )
      [item invalidateAlignment];
}

- (void) setAlignOffset :(NSPoint)offset
{
   NSAssert( NO, @"MyMovie doesn't implement setAlignOffset" );
}

- (void) imageSelectionChanged
{
   NSEnumerator* list = [_images objectEnumerator];
   id item;
   int selection_count;
   
   // Recount the selection
   selection_count = 0;
   while ( (item = [list nextObject]) != nil )
      if ( [item getSelectionState] > 0 )
         selection_count ++;
   
   // Update the state
   if ( selection_count == 0 )
      [self setSelectionState:NSOffState];
   else if ( selection_count == [self numberOfImages] )
      [self setSelectionState:NSOnState];
   else
      [self setSelectionState:NSMixedState];
}

@end
