

// PKG_ADD: autoload ("GSL_OCTAVE_NAME", which ("gsl_sf"));
DEFUN_DLD(GSL_OCTAVE_NAME, args, nargout, "\
  -*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{z} =} GSL_OCTAVE_NAME (@var{arg1}, @var{arg2})\n\
@deftypefnx {Loadable Function} {[@var{z}, @var{err}] =} GSL_OCTAVE_NAME (@dots{})\n\
\n\
GSL_FUNC_DOCSTRING
\n\
@var{err} contains an estimate of the absolute error in the value @var{z}.\n\
\n\
This function is from the GNU Scientific Library,\n\
see @url{http://www.gnu.org/software/gsl/} for documentation.\n\
DEPRECATION_WARNING
@end deftypefn\n")
{
#ifdef HAVE_GSL_FUNC

  gsl_set_error_handler (octave_gsl_errorhandler);

  // Expected number of input arguments
  const int nb_args = 2;

  // Check the actual number of input arguments
  if (args.length() != nb_args)
    {
      print_usage ();
      return octave_value ();
    }

  // Check that all input arguments are real
  for (int i = 0; i < nb_args; i++)
    if (! args(i).is_real_type ())
      {
        error ("Input argument #%d has a complex value. "
               "A real value was expected.", i + 1);
        print_usage ();
        return octave_value ();
      }

  // Get input arguments as NDArray objects
  NDArray arg1 = args(0).array_value();
  NDArray arg2 = args(1).array_value();

  // Get the dimensions of the arrays
  dim_vector dim (1, 1);
  octave_idx_type numel = 1;
  bool conformant = true;
  bool arg1_scalar = check_arg_dim<NDArray> (arg1, dim, numel, conformant);
  bool arg2_scalar = check_arg_dim<NDArray> (arg2, dim, numel, conformant);

  // Error in case of non-conformant arguments
  if (! conformant)
    {
      error ("Non-scalar input arguments must all have the same size.");
      return octave_value ();
    }

  // Create one separate index for each argument
  octave_idx_type i1 = 0;
  octave_idx_type i2 = 0;

  // Create one separate increment for each argument
  octave_idx_type inc1 = (arg1_scalar ? 0 : 1);
  octave_idx_type inc2 = (arg2_scalar ? 0 : 1);

  // Get pointers to the data of the input arrays
  const double *arg1_data = arg1.data ();
  const double *arg2_data = arg2.data ();

  // Array of output values
  NDArray y (dim);

  // GSL structure that will hold each individual result
  gsl_sf_result result;

  if (nargout < 2)
    {
      // Run the calculation
      for (octave_idx_type i = 0; i < numel; i++, i1 += inc1, i2 += inc2)
        {
          // Convert argument #1 to int
          double t1 = arg1_data[i1];
          int x1 = static_cast<int> (t1);
          if ((static_cast<double> (x1)) != t1)
            {
              y.xelem(i) = lo_ieee_nan_value ();
              continue;
            }

          GSL_FUNC_NAME (x1, arg2_data[i2], &result);
          y.xelem(i) = result.val;
        }

      return octave_value (y);
    }
  else
    {
      // Array of error estimates
      NDArray err (dim);

      // Run the calculation
      for (octave_idx_type i = 0; i < numel; i++, i1 += inc1, i2 += inc2)
        {
          // Convert argument #1 to int
          double t1 = arg1_data[i1];
          int x1 = static_cast<int> (t1);
          if ((static_cast<double> (x1)) != t1)
            {
              y.xelem(i) = lo_ieee_nan_value ();
              err.xelem(i) = lo_ieee_nan_value ();
              continue;
            }

          GSL_FUNC_NAME (x1, arg2_data[i2], &result);
          y.xelem(i) = result.val;
          err.xelem(i) = result.err;
        }

      octave_value_list retval (2);
      retval(0) = octave_value (y);
      retval(1) = octave_value (err);
      return retval;
    }

#else // HAVE_GSL_FUNC undefined

  error ("GSL function GSL_FUNC_NAME was found missing when "
         "the gsl package for octave was built.");
  return octave_value ();

#endif // HAVE_GSL_FUNC
}
