///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_VECTORN_H
#define __OVITO_VECTORN_H

#include <base/Base.h>
#include "Vector3.h"

namespace Base {

/**
 * \brief A vector with an abitrary but constant number of components.
 * 
 * \author Alexander Stukowski
 * \sa Vector_3, Vector_2
 */
template<typename T, std::size_t N>
class VectorN
{
public:
	/// The components of the vector.
	T v[N];

	/////////////////////////////// Constructors /////////////////////////////////

	/// \brief Constructs a vector without initializing its component.
	/// \note All components are left uninitialized by this constructor and will therefore have a random value!
	VectorN() {}

	/// \brief Initializes all components of the vector with the given value.
	/// \param val The value that is assigned to all components of the vector.
	explicit VectorN(T val) { for(size_t k=0; k<N; k++) v[k] = val; }

	/// \brief Initializes the vector to the null vector.
	/// All vector components are set to zero.
    VectorN(NullVector NULL_VECTOR) { for(int k=0; k<N; k++) v[k] = (T)0; }

	/// \brief Initializes the components of the vector with the values in the given array.
    VectorN(T val[N]) { for(size_t k=0; k<N; k++) v[k] = val[k]; }

	/// \brief Initializes the four components of the vector with the given values.
	/// \note This constructor may only be used if the vector template class has exactly four components.
    VectorN(T x, T y, T z, T w) { 
    	OVITO_ASSERT_MSG(N==4, "VectorN constructor", "This constructor may only be used for the Vector4 class."); 
    	v[0] = x; v[1] = y; v[2] = z; v[3] = w;
    }

    ///////////////////////////// Component access ///////////////////////////////

	/// \brief Returns a reference to the i-th component of the vector.
	/// \param i The index specifying the component to return.
	/// \return A reference to the i-th component that can be used to change the component's value. 
	T& operator[](size_t i) {
		OVITO_ASSERT(i>=0 && i<size());
		return v[i]; 
	}

	/// \brief Returns a reference to the i-th component of the vector.
	/// \param i The index specifying the component to return.
	/// \return The i-th component of the vector. 
	const T& operator[](size_t i) const {
		OVITO_ASSERT(i>=0 && i<size());
		return v[i]; 
	}

	/// \brief Returns a reference to the i-th component of the vector.
	/// \param i The index specifying the component to return.
	/// \return A reference to the i-th component that can be used to change the component's value. 
	T& operator()(size_t i) {
		OVITO_ASSERT(i>=0 && i<size());
		return v[i]; 
	}

	/// \brief Returns a reference to the i-th component of the vector.
	/// \param i The index specifying the component to return.
	/// \return The i-th component of the vector. 
	const T& operator()(size_t i) const {
		OVITO_ASSERT(i>=0 && i<size());
		return v[i]; 
	}
	
	/// \brief Returns a pointer to the first element of the vector.
	/// \sa constData() 
	T* data() { 		
		return v;
	}

	/// \brief Returns a pointer to the first element of the vector for read-only access.
	/// \sa data()
	const T* constData() const {
		return v;
	}

    /////////////////////////////// Unary operators //////////////////////////////

	/// \brief Returns the inverse of the vector.
	/// \return A vector with negated components.
	VectorN<T, N> operator-() const { 
		VectorN<T, N> n; 
        for(size_t k=0; k<N; k++) n.v[k] = -v[k];
		return n;
	}

	///////////////////////////// Assignment operators ///////////////////////////

	/// \brief Adds another vector to this vector and stores the result in this vector.
	VectorN<T, N>& operator+=(const VectorN<T, N>& w) { for(size_t k=0; k<N; k++) v[k] += w.v[k]; return *this; }

	/// \brief Substracts another vector from this vector and stores the result in this vector.
	VectorN<T, N>& operator-=(const VectorN<T, N>& w) { for(size_t k=0; k<N; k++) v[k] -= w.v[k]; return *this; }

	/// \brief Multplies each component of the vector with a scalar value and stores the result in this vector.
	VectorN<T, N>& operator*=(T s) { for(size_t k=0; k<N; k++) v[k] *= s; return *this; }

	/// \brief Divides each component of the vector by a scalar value and stores the result in this vector.
	VectorN<T, N>& operator/=(T s) { for(size_t k=0; k<N; k++) v[k] /= s; return *this; }

	////////////////////////////////// Comparison ////////////////////////////////

	/// \brief Compares two vector for equality.
	/// \return true if each of the components are equal; false otherwise.
	bool operator==(const VectorN<T, N>& w) const { 
		for(size_t k=0; k<N; k++) if(v[k] != w.v[k]) return false;
		return true;
	}

	/// \brief Compares two vector for inequality.
	/// \return true if any of the components are not equal; false if all are equal.
	bool operator!=(const VectorN<T, N>& w) const { 
		for(size_t k=0; k<N; k++) if(v[k] != w.v[k]) return true;
		return false;
	}

	/// \brief Checks whether the vector is the null vector, i.e. all components are zero.
	/// \return true if all of the components are zero; false otherwise
	bool operator==(const NullVector& NULL_VECTOR) const { 
		for(size_t k=0; k<N; k++) if(v[k] != 0) return false; 
		return true;
	}

	/// \brief Checks whether the vector is not a null vector, i.e. any of the components is nonzero.
	/// \return true if any of the components is nonzero; false if this is the null vector otherwise
	bool operator!=(const NullVector& NULL_VECTOR) const { 
		for(size_t k=0; k<N; k++) if(v[k] != 0) return true;
		return false;
	}

	/// \brief Checks whether two vectors are equal within a given tolerance.
	/// \param w The vector that should be compared to this vector.
	/// \param tolerance A non-negative threshold for the equality test. The two vectors are considered equal when 
	///        the component-wise differences are all smaller than this tolerance value.
	/// \return true if this vector is equal to the given vector within the given tolerance.
	bool Equals(const VectorN<T, N>& w, T tolerance) const { 
		for(size_t k=0; k<N; k++) if(abs(v[k] - w.v[k]) > tolerance) return false;
		return true;
	}

	/////////////////////////////// Binary operators /////////////////////////////

	/// \brief Computes the sum of two vectors.
	/// \return The sum of two vectors.
	VectorN<T, N> operator+(const VectorN<T, N>& w) const { 
        VectorN<T, N> r(*this);
        r += w;
		return r;
	}

	/// \brief Computes the difference of two vectors.
	/// \return The difference of two vectors.
	VectorN<T, N> operator-(const VectorN<T, N>& v) const { 
        VectorN<T, N> r(*this);
        r -= v;
		return r;
	}

	/// \brief Computes the product of a vector and a scalar value. All
	///        components of the vector are multiplied by the scalar.
	VectorN<T, N> operator*(T s) const { 
        VectorN<T, N> r(*this);
        r *= s;
		return r;
	}

	/// \brief Computes the division of a vector by a scalar value. All
	///        components of the vector are divided by the scalar.
	VectorN<T, N> operator/(T s) const { 
        VectorN<T, N> r(*this);
        r /= s;
		return r;
	}

	/// \brief Returns the scalar (inner or dot product) of two vectors.
	T operator*(const VectorN<T, N>& w) const { 
		T s(0);
		for(size_t k=0; k<N; k++) s += v[k] * w.v[k];
		return s;
	}

	///////////////////////////////// Information ////////////////////////////////
	
	/// \brief Returns the number of components in this vector (the constant N).
	size_t size() const { return N; }

	/// \brief Returns the dimension of this vector (the constant N).
	size_t dimension() const { return N; }

	/// \brief Returns a string representation of this vector.
	QString toString() const { QString s; for(size_t k=0; k<N; k++) { s += QString::number(v[k]); s += QChar(' '); } return s; }
};

/// \brief Computes the scalar product of two vectors.
template<typename T, int N>
inline T DotProduct(const VectorN<T, N>& a, const VectorN<T, N>& b) { 
	return a * b;
}

/// \brief Returns the squared length of a vector.
template<typename T, int N>
inline T LengthSquared(const VectorN<T, N>& a) {
	return a * a;
}

/// \brief Returns the length of a vector.
template<typename T, int N>
inline T Length(const VectorN<T, N>& a) {
	return (T)sqrt(LengthSquared(a));
}

/// \brief Returns the vector divided by its length.
template<typename T, int N>
inline VectorN<T, N> Normalize(const VectorN<T, N>& a) {
	OVITO_ASSERT(a != NULL_VECTOR);
	return a / Length(a);
}

/// \brief Writes the vector to a text output stream.
template<typename T, int N>
inline std::ostream& operator<<(std::ostream& os, const VectorN<T, N>& w) {
	for(size_t k=0; k<N; k++) {
		os << w[k];
		if(k != N - 1) os << ' ';
	}
	return os;
}

/// \brief Writes the vector to a debug stream.
template<typename T, int N>
inline QDebug operator<<(QDebug dbg, const VectorN<T, N>& w)
{
	for(size_t k=0; k<N; k++)
		dbg.space() << w[k];
	return dbg.space();
}

/// \brief Writes a vector to a binary output stream.
template<typename T, int N>
inline SaveStream& operator<<(SaveStream& stream, const VectorN<T, N>& w)
{
	for(size_t k=0; k<N; k++)
		stream << w[k];
	return stream;
}

/// \brief Reads a vector from a binary input stream.
template<typename T, int N>
inline LoadStream& operator>>(LoadStream& stream, VectorN<T, N>& w)
{
	for(size_t k=0; k<N; k++)
		stream >> w[k];
	return stream;
}

/** 
 * \fn typedef Vector4
 * \brief Template class instance of the VectorN template used for floating-point vectors. 
 */
typedef VectorN<FloatType, 4> Vector4;

/// Writes the vector to an output stream.
inline std::ostream& operator<<(std::ostream& os, const Vector4& w) {	
	return os << w[0] << ' ' << w[1] << ' ' << w[2] << ' ' << w[3];
}

/// Writes a vector to an output stream.
inline QDataStream& operator<<(QDataStream& stream, const Vector4& w)
{
	return stream << w[0] << w[1] << w[2] << w[3];
}

/// Reads a vector from an input stream.
inline QDataStream& operator>>(QDataStream& stream, Vector4& w)
{
	return stream >> w[0] >> w[1] >> w[2] >> w[3];
}

};	// End of namespace

Q_DECLARE_METATYPE(Base::Vector4)
Q_DECLARE_TYPEINFO(Base::Vector4, Q_PRIMITIVE_TYPE);

#endif // __OVITO_VECTORN_H
