///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting_gui/ScriptingGUI.h>
#include <core/utilities/PathManager.h>
#include <core/undo/UndoManager.h>
#include <scripting/engine/ScriptEngine.h>
#include "ScriptEditor.h"

namespace Scripting {

using namespace boost::python;

/******************************************************************************
* Constructs the editor frame.
******************************************************************************/
ScriptEditor::ScriptEditor(QWidget* parent) : QMainWindow(parent, Qt::Dialog | Qt::WindowMinMaxButtonsHint)
{
	// Create the central editor component.
	_scriptInput = new QsciScintilla();
	_scriptInput->setLexer(new QsciLexerPython(_scriptInput));
	_scriptInput->setMarginLineNumbers(1, true);
	setCentralWidget(_scriptInput);

	// Create the output pane.
	_outputWindow = new QTextEdit();
	_outputWindow->setReadOnly(true);
	QDockWidget* outputDockWidget = new QDockWidget(tr("Script Output"), this);
	outputDockWidget->setObjectName("ScriptOutput");
	outputDockWidget->setWidget(_outputWindow);
	outputDockWidget->setFeatures(QDockWidget::DockWidgetClosable|QDockWidget::DockWidgetMovable);
	addDockWidget(Qt::BottomDockWidgetArea, outputDockWidget);

	// Create menu bar
	QMenuBar* menuBar = new QMenuBar(this);

	QMenu* scriptMenu = menuBar->addMenu(tr("&File"));
	scriptMenu->addAction(tr("&Open"), this, SLOT(onOpen()));
    scriptMenu->addAction(tr("&Save"), this, SLOT(onSave()));
    scriptMenu->addAction(tr("Save &as"), this, SLOT(onSaveAs()));
    scriptMenu->addSeparator();
    scriptMenu->addAction(tr("&Run"), this, SLOT(onRun()), Qt::CTRL + Qt::Key_E);
    scriptMenu->addSeparator();
    scriptMenu->addAction(tr("&Close"), this, SLOT(onCloseEditor()));
	setMenuBar(menuBar);

	// Make the input window active.
	_scriptInput->setFocus();

	// Use a default window size.
	resize(640, 480);

	// Restore window layout.
	QSettings settings;
	settings.beginGroup("scripting/editor");
	QVariant state = settings.value("state");
	if(state.canConvert<QByteArray>())
		restoreState(state.toByteArray());
}

/******************************************************************************
* Handles the "Run" command event.
******************************************************************************/
void ScriptEditor::onRun()
{
	executeScript();
}

/******************************************************************************
* Executes the current script.
******************************************************************************/
bool ScriptEditor::executeScript()
{
	try {
		// Clear output window first.
		_outputWindow->clear();

		ScriptEngine engine;
		connect(&engine, SIGNAL(scriptOutput(const QString&)), this, SLOT(writeOutput(const QString&)));
		connect(&engine, SIGNAL(scriptError(const QString&)), this, SLOT(writeError(const QString&)));

		UNDO_MANAGER.beginCompoundOperation(tr("Script Actions"));
		try {
#ifdef Q_WS_WIN
			QByteArray scriptText = _scriptInput->text().toAscii();
			engine.executeScript(scriptText.replace("\r\n", "\n"));
#else
			engine.executeScript(_scriptInput->text());
#endif
		}
		catch(...) {
			UNDO_MANAGER.endCompoundOperation();
			throw;
		}
		UNDO_MANAGER.endCompoundOperation();

		return true;

	}
	catch(const Exception& ex) {
		ex.showError();
		return false;
	}
}

/******************************************************************************
* Writes some text to the output window.
******************************************************************************/
void ScriptEditor::writeOutput(const QString& str)
{
	_outputWindow->append(str);
}

/******************************************************************************
* Writes an error text to the output window.
******************************************************************************/
void ScriptEditor::writeError(const QString& str)
{
	_outputWindow->append(str);
}

/******************************************************************************
* Handles the "Close" command event.
******************************************************************************/
void ScriptEditor::onCloseEditor()
{
	close();
}

/******************************************************************************
* Handles the "Open" command event.
******************************************************************************/
void ScriptEditor::onOpen()
{
	loadFile();
}

/******************************************************************************
* Handles the "Save" command event.
******************************************************************************/
void ScriptEditor::onSave()
{
	fileSave();
}

/******************************************************************************
* Handles the "SaveAs" command event.
******************************************************************************/
void ScriptEditor::onSaveAs()
{
	fileSaveAs();
}

/******************************************************************************
* Is called when the user closes the window.
******************************************************************************/
void ScriptEditor::closeEvent(QCloseEvent* event)
{
	// Save window layout.
	QSettings settings;
	settings.beginGroup("scripting/editor");
	settings.setValue("state", saveState());
}

/******************************************************************************
* This is the implementation of the "Save" action.
* Returns true, if the script has been saved.
******************************************************************************/
bool ScriptEditor::fileSave()
{
	// Ask the user for a filename if there is no one set.
	if(filePath().isEmpty()) {
		return fileSaveAs();
	}

#if QTOK
	// Save script to file.
	try {
		wxFileOutputStream fileStream(GetFilePath());
	    if(!fileStream.Ok())
			throw Exception(wxString::Format(_("Failed to open file '%s' for writing."), GetFilePath().c_str()));
		wxTextOutputStream textStream(fileStream);
		textStream.WriteString(scriptInput->GetValue());

		if(!fileStream.Close())
			throw Exception(wxString::Format(_("Failed to close file '%s'."), GetFilePath().c_str()));
	}
	catch(const Exception& ex) {
		ex.showError();
		return false;
	}

	// Clear dirty flag.
	scriptInput->SetModified(false);
#endif
	return true;
}

/******************************************************************************
* This is the implementation of the "Save As" action.
* Returns true, if the script has been saved.
******************************************************************************/
bool ScriptEditor::fileSaveAs(QString filename)
{
#if QTOK
	if(filename.IsEmpty()) {
		wxString defaultPath;
		wxString defaultFile;
		if(GetFilePath().IsEmpty()) {
			defaultPath = PATH_MANAGER.GetScriptsDirectory().GetPath();
		}
		else {
			wxFileName fn(GetFilePath());
			defaultPath = fn.GetPath();
			defaultFile = fn.GetFullName();
		}
		filename = wxFileSelector(_("Save Script"), defaultPath, defaultFile,
			_T("py"),
			_("Python Script Files (*.py)|*.py|All Files|*"),
			wxFD_SAVE|wxFD_OVERWRITE_PROMPT, this);
        if(filename.IsEmpty())
			return false;
	}
    SetFilePath(filename);
#endif
    return fileSave();
}

/******************************************************************************
* If the script has been changed this will ask the user if he wants
* to save the changes.
* Returns false if the operation has been canceled by the user.
******************************************************************************/
bool ScriptEditor::askForSaveChanges()
{
	if(!_scriptInput->isModified())
		return true;

	QMessageBox::StandardButton result = QMessageBox::question(NULL, tr("Save changes"),
		tr("he current script has been modified. Do you want to save the changes?"),
		QMessageBox::Yes|QMessageBox::No|QMessageBox::Cancel, QMessageBox::Cancel);
	if(result == QMessageBox::Cancel)
		return false; // operation canceled by user
	else if(result == QMessageBox::No)
		return true; // go on without saving
	else {
		// Save scene
        return fileSave();
	}
}

/******************************************************************************
* Loads the given script file.
* Returns true if the file has been successfully loaded.
******************************************************************************/
bool ScriptEditor::loadFile(QString filename, bool dontAskSaveOld)
{
	if(!dontAskSaveOld && !askForSaveChanges())
		return false;

	if(filename.isEmpty()) {
		QString defaultPath;
		QSettings settings;
		if(filePath().isEmpty()) {
			defaultPath = settings.value("scripting/last_scriptfile_dir", PATH_MANAGER.scriptsDirectory()).toString();
		}
		else {
			defaultPath = filePath();
		}

		QFileDialog dialog(0, tr("Load Script"));
		dialog.setFilter(tr("Python Script Files (*.py);;All Files (*)"));
		dialog.setAcceptMode(QFileDialog::AcceptOpen);
		dialog.setFileMode(QFileDialog::ExistingFile);
		dialog.setDirectory(defaultPath);
	    if(!dialog.exec())
			return false;

		if(dialog.selectedFiles().isEmpty()) return false;
		filename = dialog.selectedFiles().front();

		settings.setValue("scripting/last_scriptfile_dir", dialog.directory().absolutePath());
	}

	try {
		QFile fileStream(filename);
		if(!fileStream.open(QIODevice::ReadOnly))
			throw Exception(tr("Failed to open script file '%1' for reading.").arg(filename));

		QApplication::setOverrideCursor(Qt::WaitCursor);
		_scriptInput->setText(QTextStream(&fileStream).readAll());
		_scriptInput->setModified(false);
		QApplication::restoreOverrideCursor();

		setFilePath(filename);
		setWindowModified(false);
		return true;
	}
	catch(const Exception& ex) {
		ex.showError();
		return false;
	}
}

};
