///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file CurrentSelectionProxy.h 
 * \brief Contains the definition of the Core::CurrentSelectionProxy class. 
 */

#ifndef __OVITO_CURRENT_SELECTION_PROXY_H
#define __OVITO_CURRENT_SELECTION_PROXY_H

#include <core/Core.h>
#include <core/scene/SelectionSet.h>

namespace Core {


/**
 * \brief Proxy for the current SelectionSet in the current DataSet.
 * 
 * The nodes contained in the current selection set of the current dataset will be
 * mirrored in this selection set proxy.
 * 
 * \note This is an internal class that is not for public use.
 * 
 * \author Alexander Stukowski
 * \sa DataSetManager::currentSelection()
 */
class CORE_DLLEXPORT CurrentSelectionProxy : public SelectionSet
{
public:
	/// Default constructor.
	CurrentSelectionProxy(bool isLoading = false);

	/// Returns the number of scene nodes in the selection set.
	virtual int count() const;

	/// Returns a scene node from the selection set.
    virtual SceneNode* node(int index) const;

	/// Returns true if the given scene node is part of the selection set.
	virtual bool contains(SceneNode* node) const;

	/// Adds a scene node to this selection set. 
    virtual void add(SceneNode* node);

	/// Adds multiple scene nodes to this selection set. 
    virtual void addAll(const QVector<SceneNode*>& nodes);

	/// Removes a scene node from this selection set. 
    virtual void remove(SceneNode* node);

	/// Clears the selection.
	virtual void clear();

	/// Returns the bounding box that includes all selected nodes.
	virtual Box3 boundingBox(TimeTicks time);

	/// Returns all nodes that are selected.
	virtual const QVector<SceneNode*>& nodes() const;

	/// Sets the contents of the selection set.
	virtual void setNodes(const QVector<SceneNode*>& nodes);
	
	/// Resets the selection set to contain only a single node. 
	virtual void setNode(SceneNode* node);
	
	/// Gets the selection set which this proxy encapsulates.
	SelectionSet* currentSelectionSet() const { return _selectionSet; }

	/// Sets the selection set which this proxy encapsulates.
	void setCurrentSelectionSet(SelectionSet* set) { _selectionSet = set; onChanged(); }

protected:

	/// From RefMaker.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Is called when the selection set has changed.
	/// Raises the SelectionChanged event for the DataSetManager.
	void onChanged();
	
Q_SIGNALS:

	/// This is only an internal signal.
	void internalSelectionChanged();
	
protected Q_SLOTS:

	/// Is called after the selection set has changed multiple times.
	void onInternalSelectionChanged();
	
private:

	/// Holds the references to the current selection set in the current data set.
	ReferenceField<SelectionSet> _selectionSet;
	
	/// Indicates that there is a pending change event in the event queue.
	bool changeEventInQueue;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CurrentSelectionProxy)
	DECLARE_REFERENCE_FIELD(_selectionSet)
};


};

#endif // __OVITO_CURRENT_SELECTION_PROXY_H
