/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.lang.System;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.htmlparser.Node;
import org.htmlparser.NodeFilter;
import org.htmlparser.Parser;
import org.htmlparser.filters.HasAttributeFilter;
import org.htmlparser.filters.OrFilter;
import org.htmlparser.filters.TagNameFilter;
import org.htmlparser.nodes.TextNode;
import org.htmlparser.util.NodeList;
import org.htmlparser.util.ParserException;
import org.videolan.vlma.common.Configuration;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.IVlOrderGiver;
import org.videolan.vlma.common.VlMonostate;
import org.videolan.vlma.common.VlServer;
import org.videolan.vlma.common.VlServerReal;
import org.videolan.vlma.common.adapters.IVlAdapter;
import org.videolan.vlma.common.adapters.VlAdapter;
import org.videolan.vlma.common.adapters.VlDVBS;
import org.videolan.vlma.common.adapters.VlDVBT;
import org.videolan.vlma.common.exceptions.AdapterAlreadyExistsException;
import org.videolan.vlma.common.exceptions.AdapterDoesNotExistException;
import org.videolan.vlma.common.exceptions.AdapterParameterDoesNotExistException;
import org.videolan.vlma.common.exceptions.MediaAlreadyExistsException;
import org.videolan.vlma.common.exceptions.MediaDoesNotExistException;
import org.videolan.vlma.common.exceptions.SatelliteAlreadyExistsException;
import org.videolan.vlma.common.exceptions.SatelliteDoesNotExistException;
import org.videolan.vlma.common.exceptions.ServerAlreadyExistsException;
import org.videolan.vlma.common.exceptions.ServerDoesNotExistException;
import org.videolan.vlma.common.medias.IVlMedia;
import org.videolan.vlma.common.medias.VlSatChannel;
import org.videolan.vlma.common.medias.VlSatellite;
import org.videolan.vlma.common.medias.VlTNTChannel;
import org.videolan.vlma.common.medias.VlFilesChannel;
import org.videolan.vlma.common.medias.VlMedia;
import org.videolan.vlma.common.orders.VlOrder;
import org.videolan.vlma.common.programs.IVlProgram;
import org.videolan.vlma.common.programs.VlIpBank;
import org.videolan.vlma.common.programs.VlProgram;

import com.thoughtworks.xstream.XStream;

/**
 * This is an implementation of the IVlData interface. This class is used to
 * contain the data and make them available through remoting.
 * 
 * @author SylV
 * @author vivi
 */
public class VlData implements IVlData {

    private static final Logger logger = Logger.getLogger(VlData.class);
    
    private List<VlServer> servers;
    private List<IVlMedia> medias;    
    private HashMap<Integer, List<VlOrder>> orders;    
    private VlIpBank ipBank;
    private List<VlSatellite> satellites;
    
    private VlOrderGiver orderGiver;
    private VlOrderMonitoring orderMonitoring;
    private VlServerMonitoring serverMonitoring;

    private static XStream xstream;
    
    static {
        xstream = new XStream();
        xstream.alias("satellite", VlSatellite.class);
        xstream.alias("satChannel", VlSatChannel.class);
        xstream.alias("server", VlServer.class);
        xstream.alias("server", VlServerReal.class);
        xstream.alias("dvb-t", VlDVBT.class);
        xstream.alias("dvb-s", VlDVBS.class);
        xstream.alias("program", VlProgram.class);
        xstream.alias("tntChannel", VlTNTChannel.class);
        xstream.alias("configuration", Configuration.class);
        xstream.setMode(XStream.XPATH_ABSOLUTE_REFERENCES);
    }
    
    public VlData() {
        servers = new ArrayList<VlServer>();
        medias = new ArrayList<IVlMedia>(200);
        ipBank = new VlIpBank(this);
        orders = new HashMap<Integer, List<VlOrder>>();
        orderGiver = new VlOrderGiver(this);
        satellites = new ArrayList<VlSatellite>();
    }
    
    synchronized public Configuration getConfiguration() {
        return Configuration.getInstance();
    }
    
    synchronized public List<VlSatellite> getSatellites() {
        return satellites;
    }
    
    synchronized public VlSatellite getSatellite(int satellite)
      throws SatelliteDoesNotExistException {
        for (VlSatellite s : satellites) {
            if (s.hashCode() == satellite) {
                return s;
            }
        }
        throw new SatelliteDoesNotExistException("Satellite " + satellite + " doesn't exist.");
    }
    
    synchronized public void clearSatelliteCoverages(int satellite)
      throws SatelliteDoesNotExistException {
        VlSatellite s = getSatellite(satellite);
        s.getCoverages().clear();
    }
    
    synchronized public void addSatelliteCoverage(int satellite, String coverage)
      throws SatelliteDoesNotExistException {
        coverage = coverage.trim();        
        VlSatellite s = getSatellite(satellite);
        for (String c : coverage.split("\n")) {
            s.getCoverages().add(c.trim());
        }
    }

    
    synchronized private void addSatelliteToTheList(VlSatellite s, String name)
    {
        /**
         * This function adds the satellite to list of satellites of this VLData
         * object.
         */
        satellites.add(s);
        logger.log(Level.DEBUG, "Satellite " + name + " added.");
    }
    
    
    synchronized public int addSatellite(String name) 
      throws SatelliteAlreadyExistsException {
        VlSatellite s = new VlSatellite(name);
        if (satellites != null) {
            if (satellites.contains(s)) {
                throw new SatelliteAlreadyExistsException("Datellite " + name + " already exists.");
            } else {
                addSatelliteToTheList(s, name);
                return s.hashCode();
            }
        } else {
            addSatelliteToTheList(s, name);
            return s.hashCode();
        }
    }
    
    public List<String> getSatelliteCoverages(int satellite)
      throws SatelliteDoesNotExistException {
        VlSatellite s = getSatellite(satellite);
        return s.getCoverages();
    }
    
    public void removeSatellite(int satellite)
      throws SatelliteDoesNotExistException {
        VlSatellite s = getSatellite(satellite);
        satellites.remove(s);
    }

    synchronized public List<VlServer> getServers() {
        return servers;
    }

    synchronized public VlServer getServer(int server)
            throws ServerDoesNotExistException {
        for (VlServer s : servers) {
            if (s.hashCode() == server) {
                return s;
            }
        }
        throw new ServerDoesNotExistException("Server " + server + " doesn't exist.");
    }
    
    synchronized public VlServer getServer(String server)
    throws ServerDoesNotExistException {
        for (VlServer s : servers) {
            logger.log(Level.DEBUG, "Server-test : " + s.getName());
            logger.log(Level.DEBUG, "Server : " + server);
            if (s.getName().equals(server)) {
                return s;
            }
        }
        throw new ServerDoesNotExistException("Server " + server + " doesn't exist.");
    }

    synchronized public void setServerIp(int server, InetAddress newIp)
            throws ServerDoesNotExistException {
        VlServer s = getServer(server); 
        s.setIp(newIp);
    }

    synchronized public int addServer(String name) throws ServerAlreadyExistsException {
        VlServer s;
        
        try {
            Constructor myConstructor = VlMonostate.serverClass.getConstructor(new Class[] {String.class, InetAddress.class});
            s = (VlServer) myConstructor.newInstance(new Object[] {name, null});
        }
        catch (Exception e) {
            throw new ServerAlreadyExistsException("Serveur class invalid.");
        }
        
        /*
         * if( Monostate.g_FakeDaemon ) { s =
         * Monostate.g_ServerClass.newInstance(name, null); } else { s = new
         * VlServerReal(name, null); }
         */
        
         
        if (servers.contains(s))
        {
            throw new ServerAlreadyExistsException("Server " + name + " already exists.");
        } else {
            servers.add(s);
            logger.log(Level.DEBUG, "Serveur " + name + " added.");
            return s.hashCode(); 
        }
    }

    synchronized public void removeServer(int server)
            throws ServerDoesNotExistException {
        
        VlServer sv = this.getServer(server);
        if (servers.remove(sv))
        {
            logger.log(Level.DEBUG, "Serveur " + server + " removed.");
        }
        else
        {
            throw new ServerDoesNotExistException("Server " + server + " doesn't exist.");
        }
    }

    synchronized public Map<String, IVlAdapter> getAdapters(int server)
            throws ServerDoesNotExistException {
        VlServer s = this.getServer(server);
        return s.getAdapters();
    }

    synchronized public IVlAdapter getAdapter(int server, String name)
            throws ServerDoesNotExistException, AdapterDoesNotExistException {
        VlServer s = this.getServer(server);
        HashMap<String, IVlAdapter> m = (HashMap<String, IVlAdapter>)s.getAdapters();
        IVlAdapter a = (IVlAdapter) m.get(name);

        if (a == null) {
            throw new AdapterDoesNotExistException("Adapter " + name + " doesn't exist on server " + server + ".");
        }
        return a;
    }

    synchronized public void setAdapterName(int server, String name,
            String newName) throws AdapterDoesNotExistException,
            AdapterAlreadyExistsException {
        VlServer s = this.getServer(server);
        Map<String, IVlAdapter> m = s.getAdapters();
        IVlAdapter a = (IVlAdapter) m.get(name);
        if (a == null) {
            throw new AdapterDoesNotExistException("Adapter " + name + " doesn't exist on server " + server + ".");
        }
        if (m.containsKey(newName)) {
            throw new AdapterAlreadyExistsException("Adapter " + newName + " already exists in server " + server + ".");
        }
        a.setName(newName);
        m.put(newName, a);
        m.remove(name);
    }

    synchronized public void setAdapterParameter(int server, String name,
            String parameter, String value)
            throws AdapterDoesNotExistException, AdapterParameterDoesNotExistException {
        VlServer s = this.getServer(server);
        Map m = s.getAdapters();
        IVlAdapter a = (IVlAdapter) m.get(name);
        if (a == null) {
            throw new AdapterDoesNotExistException("Adapter " + name + " doesn't exist in server " + server + ".");
        }
        a.setParameter(parameter, value, this);
    }

    @SuppressWarnings("unchecked")
    synchronized public void addAdapter(int server, String name, String type)
            throws AdapterAlreadyExistsException, IllegalAccessException,
            InstantiationException, ClassNotFoundException {
        VlServer s = this.getServer(server);
        Map<String, IVlAdapter> m = s.getAdapters();
        if (m.containsKey(name)) {
            throw new AdapterAlreadyExistsException("Adapter " + name + " already exists in server " + server + ".");
        }
        Class<VlAdapter> c = (Class<VlAdapter>) Class.forName(type);
        IVlAdapter a = c.newInstance();
        a.setName(name);
        a.setServer(s);
        m.put(name, a);
    }

    synchronized public void removeAdapter(int server, String name)
            throws AdapterDoesNotExistException {
        VlServer s = this.getServer(server);
        Map m = s.getAdapters();
        if (m.remove(name) == null) {
            throw new AdapterDoesNotExistException("Adapter " + name + " doesn't exist in server " + server + ".");
        }
    }
    
    synchronized public List<IVlMedia> getMedias() {
        return medias;
    }
    
    synchronized public IVlMedia getMedia(int media) {
        for (IVlMedia m : medias) {
            if (m.getClass().equals(VlSatChannel.class)) {
                if (((VlSatChannel) m).hashCode() == media) {
                    return m;
                }
            } else if (m.getClass().equals(VlTNTChannel.class)) {
                if (((VlTNTChannel) m).hashCode() == media) {
                    return m;
                }
            } else if (m.getClass().equals(VlFilesChannel.class) && ((VlFilesChannel) m).hashCode() == media) {
                return m;
            }
            
        }
        throw new MediaDoesNotExistException("Media " + media + " doesn't exist");
    }

    synchronized public int addMedia(IVlMedia media) {
        if (medias.contains(media)) {
            throw new MediaAlreadyExistsException("Media " + media.getName() + " already exists");
        } else {
            medias.add(media);
            logger.log(Level.DEBUG, "Media " + media.getName() + " added.");
            return media.hashCode();
        }
    }

    synchronized public void removeMedia(int media) {
        IVlMedia m = this.getMedia(media);
        if (medias.remove(m)) {
            logger.log(Level.DEBUG, "Media " + media + " removed");
        } else {
            throw new MediaDoesNotExistException("Media " + media + " doesn't exist");
        }
    }

    synchronized public void updateMediaProgram(int media,
            IVlProgram program) throws MediaDoesNotExistException {
        IVlMedia m = this.getMedia(media);
        m.setProgram(program);
        logger.log(Level.DEBUG, "New programmation for media " + media);
    }

    synchronized public VlProgram addMediaProgram(int media)
            throws MediaDoesNotExistException {
        IVlMedia m = this.getMedia(media);
        VlProgram program = new VlProgram();
        InetAddress a = ipBank.getIp();
        logger.log(Level.DEBUG, "Multicast IPv4 address for " + m.getName() + ": " + a.toString());
        program.setIp(a);
        m.setProgram(program);
        return program;        
    }
    
    synchronized public void setFilesChannelFilesList(int filesChannel, String files)
      throws MediaDoesNotExistException {
        files = files.trim();
        IVlMedia m = getMedia(filesChannel);
        ((VlFilesChannel)m).setFiles(new ArrayList<String>());
        for (String c : files.split("\n")) {
            ((VlFilesChannel)m).getFiles().add(c.trim());
        }
    }

    synchronized public void setFilesChannelServer(int filesChannel, String server)
        throws ServerDoesNotExistException {
        IVlMedia m = getMedia(filesChannel);
        ((VlFilesChannel)m).setServer(getServer(server));
    }
    
    
    /**
     * Extract data from a HTML row where, in each cell, several <br />
     * -separated `rows' can be found.
     * 
     * @param nodes
     *            the list of nodes which represent the row
     * @param column
     *            the column (beginning with 0) where to find data
     * @param row
     *            the row (beginning with 0) where to find data
     * @param maxRow
     *            the expected number of rows - 1 (in fact the highest row
     *            number)
     */
    private static String htmlCellExtract(NodeList nodes, int column, int row, int maxRow) throws ParserException {
        String result = "";
        if (maxRow == 0) {
            Node text = nodes.elementAt(column);
            result = text.toPlainTextString();            
        }
        else {
            NodeFilter brTagFilter = new TagNameFilter("br");
            NodeList breakLines = nodes.elementAt(column).getChildren().extractAllNodesThatMatch(brTagFilter);
            if (breakLines.size() == maxRow) {
                if (row == 0) {
                    Node text = breakLines.elementAt(0).getPreviousSibling();
                    while (!(text instanceof TextNode) || "".equals(((TextNode) text).getText().trim())) {                            
                        text = text.getPreviousSibling();
                        if (text == null) {
                            return "";
                        }
                    }
                    result = ((TextNode) text).getText();                
                }
                else {
                    Node text = breakLines.elementAt(row - 1).getNextSibling();
                    while (!(text instanceof TextNode) || "".equals(((TextNode) text).getText().trim())) {                            
                        text = text.getNextSibling();
                        if (text == null) {
                            return "";
                        }
                    }
                    result = ((TextNode) text).getText();    
                }
            }
        }
        result = result.replaceAll("&nbsp;", " ");
        return result.trim();
    }

    synchronized public void updateSatChannels(URL source) throws IOException {
        List<String> tvFilter = new ArrayList<String>();
        Set<String> coverages = new HashSet<String>();
        tvFilter.add("TV-DIG-CRYPT");
        tvFilter.add("TV-DIG");
        tvFilter.add("R-DIG");
        tvFilter.add("R-DIG-CRYPT");
        tvFilter.add("TV-HD");
        Set<IVlMedia> newMedias = new HashSet<IVlMedia>();
        newMedias.clear();
        logger.log(Level.DEBUG, "Downloading " + source.toString() + ".");

        try {
            Parser parser = new Parser(source.openConnection());
            NodeList rows = parser.parse(new TagNameFilter("tr"));
            logger.log(Level.DEBUG, source.toString() + " downloaded and parsed.");
            int frequency = 0;
            char polarisation = 'A';
            String coverage = "";
            NodeFilter thTagFilter = new OrFilter(new TagNameFilter("th"), new HasAttributeFilter("rowspan"));
            NodeFilter tdTagFilter = new TagNameFilter("td"); 
            for (int i = 0; i < rows.size(); ++i) {
                Node node = rows.elementAt(i);
                NodeList thCells = node.getChildren().extractAllNodesThatMatch(thTagFilter);
                NodeList tdCells = node.getChildren().extractAllNodesThatMatch(tdTagFilter);
                if (tdCells.size() == 8) {
                    if (thCells.size() == 2) {
                        frequency = (int) (Double.parseDouble(htmlCellExtract(thCells, 0, 1, 2).split(" ")[0]) * 1000);
                        polarisation = htmlCellExtract(thCells, 0, 2, 2).charAt(0);
                        coverage = htmlCellExtract(thCells, 1, 0, 0);
                    }                
                    String name = htmlCellExtract(tdCells, 1, 0, 3);
                    String category = htmlCellExtract(tdCells, 0, 0, 2);
                    String encryption = htmlCellExtract(tdCells, 2, 1, 3);
                    int symbolRate = 27500;
                    try {
                        symbolRate = (int) (Double.parseDouble(htmlCellExtract(tdCells, 3, 0, 3))) * 1000;
                    } catch (ArrayIndexOutOfBoundsException e) {
                    } catch (NumberFormatException e) {                    
                    }
                    int errorCorrection = 9;
                    try {
                        errorCorrection = Integer.parseInt((htmlCellExtract(tdCells, 3, 0, 3).substring(0, 1)));
                    } catch (ArrayIndexOutOfBoundsException e) {
                    } catch (StringIndexOutOfBoundsException e) {
                    } catch (NumberFormatException e) {                    
                    }
                    
                    int sid = 0;
                    try {
                        sid = Integer.parseInt(htmlCellExtract(tdCells, 4, 1, 3));
                    } catch (NumberFormatException e) {
                    } catch (ArrayIndexOutOfBoundsException e) {
                    }
                    String country = htmlCellExtract(tdCells, 5, 0, 3);
                    VlSatChannel ch = new VlSatChannel();
                    ch.setFrequency(frequency);
                    ch.setPolarisation(polarisation);
                    ch.setCoverage(coverage);
                    ch.setName(name);
                    ch.setCategory(category);
                    ch.setEncryption(encryption);
                    ch.setSymbolRate(symbolRate);
                    ch.setErrorCorrection(errorCorrection);
                    ch.setSid(sid);
                    ch.setCountry(country);
                    if (tvFilter.contains(ch.getCategory())) {
                        newMedias.add(ch);
                    }
                }
            }
        }
        catch (ParserException e) {
            throw new IOException();
        }
        
        Iterator j = newMedias.iterator();
        while (j.hasNext()) {
            VlSatChannel ch = (VlSatChannel) j.next();
            logger.log(Level.DEBUG, "Adding " + ch.getName());
            coverages.add(ch.getCoverage());
            if (medias.contains(ch)) {
                ch.setProgram(medias.get(medias.indexOf(ch)).getProgram());
            }
        }
        
        logger.log(Level.DEBUG, "Got " + newMedias.size() + " channels");
        Iterator k = medias.iterator();
        while (k.hasNext()) {
            VlMedia ch = (VlMedia) k.next();
            if (ch.getClass().getName() == "VlSatChannel" && coverages.contains(((VlSatChannel)ch).getCoverage()))
            {
                k.remove();
            }
        }
        medias.addAll(newMedias);
        sortMedias();
        logger.log(Level.DEBUG, "Ending analysis of " + source.toString());
    }

    public HashMap<Integer, List<VlOrder>> getOrders() {
        return orders;
    }
    
    public void giveOrders() {
        orderGiver.computeOrders();
    }
    
    public IVlOrderGiver getOrderGiver() {
        return orderGiver;
    }

    synchronized public void saveToDisk() throws FileNotFoundException,
            IOException {
        List<List> saveList = new ArrayList<List>();
        saveList.add(satellites);
        saveList.add(servers);
        saveList.add(medias);
        
        String vlmaHomePath = new String(System.getProperty("user.home") + 
                File.separator + ".vlma" + File.separator);
        
        /* Create a ".vlma" directory in the user home if it doesn't exist */
        File vlmaHome = new File(vlmaHomePath);
        if (!vlmaHome.exists()) {
            vlmaHome.mkdir();
        }
        
        FileOutputStream f = new FileOutputStream(vlmaHomePath + "data.xml");
        xstream.toXML(saveList, f);
        f.close();
        
        f = new FileOutputStream(vlmaHomePath +    "config.xml");
        
        xstream.toXML(getConfiguration(), f);
        f.close();
    }

    @SuppressWarnings({"unchecked","unchecked"})
    synchronized public void loadFromDisk() throws FileNotFoundException,
            IOException {
        String vlmaHomePath = new String(System.getProperty("user.home") + File.separator + ".vlma" + File.separator);
        
        try {
            FileInputStream f = new FileInputStream(vlmaHomePath + "data.xml");
            
            List<List> loadList;
            loadList = (List<List>) xstream.fromXML(f);
            f.close();
            
            satellites = (List<VlSatellite>) loadList.get(0);
            servers = (List<VlServer>) loadList.get(1);
            medias = (List<IVlMedia>) loadList.get(2);
        }
        catch (FileNotFoundException e) {
            logger.log(Level.WARN, "Unable to read data.xml");
        }
        try {
            FileInputStream f = new FileInputStream(vlmaHomePath + "config.xml");
            
            Configuration configuration = getConfiguration();
            configuration = (Configuration) xstream.fromXML(f);
            f.close();
        }
        catch (FileNotFoundException e) {
            logger.log(Level.WARN, "Unable to read config.xml");
        }
    }
    
    
    /**
     * This method inits the IP bank
     * 
     * @param IPsDevel
     *            true if we devel IPs must be used for the orders
     */
    synchronized public void initIPs(boolean IPsDevel) {
        try {
        ipBank.initIps(IPsDevel);
        }
        catch (UnknownHostException e) {
        }
    }

    public void setServers(List<VlServer> servers) {
        this.servers = servers;
    }
    
    public void setMedias(List<IVlMedia> medias) {
        this.medias = medias;
    }
    
    public void sortMedias()
    {
        Collections.sort(medias);
    }
    
    /*
     * For the Monitoring class
     */
    
    /* Starts the deamons */
    public void startServerMonitoringDaemon() {
        serverMonitoring.startServerMonitoringDeamon();
    }
    
    public void startOrderMonitoringDaemon() {
        orderMonitoring.startOrderMonitoringDeamon();
    }
    
    /* Starts the punctual monitoring controls */
    public void startOrderMonitoring() {
        orderMonitoring.startOrderMonitoringThread();
    }
    
    public void startCheckAllVLCs() {
        serverMonitoring.startCheckVLCThread();
    }
    
    /* Setters for spring application context */
    public void setOrderMonitoring(VlOrderMonitoring var) {
        orderMonitoring = var;
    }

    public void setServerMonitoring(VlServerMonitoring var) {
        serverMonitoring = var;
    }
    
}
